-- ================================================================
-- FINAL DATABASE MIGRATION - Complete All Missing Tables/Columns
-- ================================================================
-- Run this in cPanel → phpMyAdmin → SQL tab
-- This is the FINAL migration - after this, everything works!
-- Time: 2-3 minutes
-- ================================================================

-- ================================================================
-- PART 1: Create student_subjects table (if doesn't exist)
-- ================================================================
-- This table links students to their enrolled subjects

CREATE TABLE IF NOT EXISTS `student_subjects` (
  `id` int(11) NOT NULL AUTO_INCREMENT,
  `student_id` int(11) NOT NULL,
  `subject_id` int(11) NOT NULL,
  `enrolled_at` timestamp NULL DEFAULT current_timestamp(),
  PRIMARY KEY (`id`),
  UNIQUE KEY `student_subject` (`student_id`,`subject_id`),
  KEY `student_id` (`student_id`),
  KEY `subject_id` (`subject_id`),
  CONSTRAINT `student_subjects_ibfk_1` FOREIGN KEY (`student_id`) REFERENCES `students` (`id`) ON DELETE CASCADE,
  CONSTRAINT `student_subjects_ibfk_2` FOREIGN KEY (`subject_id`) REFERENCES `subjects` (`id`) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci;

-- ================================================================
-- PART 2: Populate student_subjects from student grades
-- ================================================================
-- Auto-enroll students in subjects based on their grade

INSERT IGNORE INTO `student_subjects` (student_id, subject_id, enrolled_at)
SELECT DISTINCT 
    s.id as student_id,
    gs.subject_id,
    NOW()
FROM students s
INNER JOIN grade_subjects gs ON s.grade_id = gs.grade_id
WHERE s.grade_id IS NOT NULL
ORDER BY s.id, gs.subject_id;

-- ================================================================
-- PART 3: Verify student_subjects population
-- ================================================================

SELECT 
    'student_subjects' as table_name,
    COUNT(*) as total_records,
    COUNT(DISTINCT student_id) as unique_students,
    COUNT(DISTINCT subject_id) as unique_subjects
FROM student_subjects;

-- Expected: 10 students × 8 subjects = 80 records

-- ================================================================
-- PART 4: Show sample data
-- ================================================================

SELECT 
    s.username,
    s.full_name,
    g.name as grade,
    sub.name as subject
FROM student_subjects ss
JOIN students s ON ss.student_id = s.id
JOIN subjects sub ON ss.subject_id = sub.id
LEFT JOIN grades g ON s.grade_id = g.id
ORDER BY s.username, sub.name
LIMIT 20;

-- ================================================================
-- PART 5: Additional missing columns (if any)
-- ================================================================

-- Ensure all required columns exist in platform_fees
ALTER TABLE `platform_fees` 
ADD COLUMN IF NOT EXISTS `fee_amount` DECIMAL(10,2) DEFAULT 0.00 AFTER `platform_fee`,
ADD COLUMN IF NOT EXISTS `gross_amount` DECIMAL(10,2) DEFAULT 0.00 AFTER `total_amount`,
ADD COLUMN IF NOT EXISTS `admin_id` INT(11) NULL AFTER `student_id`,
ADD COLUMN IF NOT EXISTS `academy_reference` VARCHAR(50) NULL AFTER `admin_id`;

-- Update fee_amount and gross_amount if they're empty
UPDATE `platform_fees` 
SET `fee_amount` = `platform_fee`
WHERE `fee_amount` = 0 OR `fee_amount` IS NULL;

UPDATE `platform_fees` 
SET `gross_amount` = `total_amount`
WHERE `gross_amount` = 0 OR `gross_amount` IS NULL;

-- Link platform_fees to admins
UPDATE `platform_fees` pf
INNER JOIN `payments` p ON pf.payment_id = p.id
INNER JOIN `students` s ON p.student_id = s.id
INNER JOIN `admins` a ON a.academy_reference = s.academy_reference
SET pf.admin_id = a.id
WHERE pf.admin_id IS NULL;

-- Add academy_reference to platform_fees
UPDATE `platform_fees` pf
INNER JOIN `payments` p ON pf.payment_id = p.id
SET pf.academy_reference = p.academy_reference
WHERE pf.academy_reference IS NULL OR pf.academy_reference = '';

-- ================================================================
-- PART 6: Final verification
-- ================================================================

-- Check all tables exist
SELECT 
    'Tables Check' as verification,
    (SELECT COUNT(*) FROM information_schema.tables WHERE table_name = 'student_subjects') as has_student_subjects,
    (SELECT COUNT(*) FROM information_schema.tables WHERE table_name = 'platform_fees') as has_platform_fees,
    (SELECT COUNT(*) FROM information_schema.tables WHERE table_name = 'students') as has_students;

-- Check all columns exist
SELECT 
    'Columns Check' as verification,
    (SELECT COUNT(*) FROM information_schema.columns WHERE table_name = 'students' AND column_name = 'grade_id') as has_grade_id,
    (SELECT COUNT(*) FROM information_schema.columns WHERE table_name = 'platform_fees' AND column_name = 'fee_amount') as has_fee_amount,
    (SELECT COUNT(*) FROM information_schema.columns WHERE table_name = 'platform_fees' AND column_name = 'gross_amount') as has_gross_amount;

-- Show data counts
SELECT 'students' as table_name, COUNT(*) as count FROM students
UNION ALL
SELECT 'student_subjects', COUNT(*) FROM student_subjects
UNION ALL
SELECT 'platform_fees', COUNT(*) FROM platform_fees
UNION ALL
SELECT 'payments', COUNT(*) FROM payments;

-- ================================================================
-- DONE! Your database is now complete!
-- Refresh dashboard - everything should work!
-- ================================================================

