# 🔧 FIX SUBJECTS TABLE - DUPLICATE ENTRY GUIDE

## 🎯 **The Problem**

Your `subjects` table has a **UNIQUE constraint on the `name` column**. This prevents importing the Eswatini curriculum because the same subject (e.g., "Mathematics") needs to exist for multiple grades.

### **Current Error:**
```
Warning: #1062 Duplicate entry 'Mathematics' for key 'name'
Warning: #1062 Duplicate entry 'English Language' for key 'name'
```

### **Why This Happens:**

```
❌ CURRENT (WRONG) DESIGN:
subjects table: UNIQUE constraint on 'name'
├─ id: 1, name: "Mathematics", code: "G4-MAT-003" ✅
└─ Try to add: name: "Mathematics", code: "G5-MAT-003" ❌ DUPLICATE ERROR
```

### **What We Need:**

```
✅ FIXED (CORRECT) DESIGN:
subjects table: UNIQUE constraint on 'code' (NOT name)
├─ id: 1, name: "Mathematics", code: "G4-MAT-003" ✅
├─ id: 2, name: "Mathematics", code: "G5-MAT-003" ✅
├─ id: 3, name: "Mathematics", code: "G6-MAT-003" ✅
└─ ... all different codes, same name allowed ✅
```

---

## 🚀 **COMPLETE FIX - STEP BY STEP**

### **STEP 1: Fix the Table Structure (5 minutes)**

Run this in phpMyAdmin:

```sql
-- Import the migration script
SOURCE database/migrations/fix_subjects_duplicates.sql;
```

**OR** Run manually:

```sql
-- 1. Backup
CREATE TABLE subjects_backup AS SELECT * FROM subjects;
CREATE TABLE grade_subject_backup AS SELECT * FROM grade_subject;

-- 2. Remove UNIQUE constraint from 'name'
ALTER TABLE subjects DROP INDEX name;

-- 3. Ensure 'code' is UNIQUE
ALTER TABLE subjects ADD UNIQUE KEY unique_code (code);

-- 4. Verify
SHOW INDEXES FROM subjects;
```

**Expected Result:**
```
✅ name: No UNIQUE constraint
✅ code: UNIQUE constraint (unique_code)
```

---

### **STEP 2: Clean Up Old Data (2 minutes)**

**Option A: Complete Clean Slate (Recommended)**

```sql
-- Remove all subjects and start fresh
SET FOREIGN_KEY_CHECKS = 0;
TRUNCATE TABLE grade_subject;
TRUNCATE TABLE subjects;
SET FOREIGN_KEY_CHECKS = 1;
```

**Option B: Keep Existing Subjects**

```sql
-- Only remove Eswatini curriculum subjects
DELETE gs FROM grade_subject gs
JOIN subjects s ON gs.subject_id = s.id
WHERE s.code LIKE 'G4-%' OR s.code LIKE 'G5-%' OR s.code LIKE 'G6-%' 
   OR s.code LIKE 'G7-%' OR s.code LIKE 'F1-%' OR s.code LIKE 'F2-%' 
   OR s.code LIKE 'F3-%' OR s.code LIKE 'F4-%' OR s.code LIKE 'F5-%';

DELETE FROM subjects
WHERE code LIKE 'G4-%' OR code LIKE 'G5-%' OR code LIKE 'G6-%' 
   OR code LIKE 'G7-%' OR code LIKE 'F1-%' OR code LIKE 'F2-%' 
   OR code LIKE 'F3-%' OR code LIKE 'F4-%' OR code LIKE 'F5-%';
```

---

### **STEP 3: Verify the Fix (1 minute)**

```sql
-- Run verification script
SOURCE database/verify_subjects_fix.sql;
```

**Expected Output:**
```
✅ CORRECT: No UNIQUE constraint on name
✅ CORRECT: UNIQUE constraint exists on code
✅ PASS: No duplicate codes
✅✅✅ ALL CHECKS PASSED - Ready for curriculum import!
```

---

### **STEP 4: Re-Import Eswatini Curriculum (2 minutes)**

```sql
-- Import the complete curriculum
SOURCE database/ESWATINI_CURRICULUM_SUBJECTS.sql;
```

**Expected Result:**
```
✅ ~200+ subjects imported
✅ No duplicate entry warnings (some may be skipped with INSERT IGNORE, but no errors)
✅ All grades linked to subjects
```

---

### **STEP 5: Verify Import Success**

```sql
-- Check total subjects
SELECT COUNT(*) as total_subjects FROM subjects;
-- Should return: 150-200+ subjects

-- Check subjects per grade
SELECT g.name, COUNT(gs.subject_id) as subject_count
FROM grades g
LEFT JOIN grade_subject gs ON g.id = gs.grade_id
GROUP BY g.name
ORDER BY g.name;

-- Expected results:
-- Grade 4: 8-12 subjects
-- Grade 5: 7-13 subjects
-- Grade 6: 9-14 subjects
-- Grade 7: 11-15 subjects
-- Form 1: 11-22 subjects
-- Form 2: 6-24 subjects
-- Form 3: 13-25 subjects
-- Form 4: 11-21 subjects
-- Form 5: 8-26 subjects

-- Check Mathematics across grades
SELECT name, code FROM subjects 
WHERE name LIKE '%Mathematics%'
ORDER BY code;

-- Should show multiple Mathematics entries:
-- Mathematics | G4-MAT-003
-- Mathematics | G5-MAT-003
-- Mathematics (Algebra...) | F2-MAT-003
-- etc.
```

---

## 📁 **Files Included**

```
✅ database/migrations/fix_subjects_duplicates.sql
   - Complete migration script
   - Includes backup, fix, verification
   - Safe to run (includes rollback)

✅ database/verify_subjects_fix.sql
   - Comprehensive verification queries
   - Checks constraints, duplicates, counts
   - Final status summary

✅ database/cleanup_and_reimport.sql
   - Options to clean old data
   - Selective or complete cleanup
   - Includes rollback scripts

✅ FIX_SUBJECTS_DUPLICATE_GUIDE.md
   - This complete guide
   - Step-by-step instructions
```

---

## 🎯 **Quick Commands (Copy-Paste)**

### **Complete Fix in 3 Commands:**

```sql
-- 1. Fix table structure
ALTER TABLE subjects DROP INDEX name;
ALTER TABLE subjects ADD UNIQUE KEY unique_code (code);

-- 2. Clean slate
SET FOREIGN_KEY_CHECKS = 0;
TRUNCATE TABLE grade_subject;
TRUNCATE TABLE subjects;
SET FOREIGN_KEY_CHECKS = 1;

-- 3. Re-import (in phpMyAdmin Import tab)
-- Upload: database/ESWATINI_CURRICULUM_SUBJECTS.sql
```

---

## ✅ **Verification Checklist**

After running the fix, verify:

- [ ] `SHOW INDEXES FROM subjects;` - 'name' has no UNIQUE
- [ ] `SHOW INDEXES FROM subjects;` - 'code' has UNIQUE (unique_code)
- [ ] `SELECT COUNT(*) FROM subjects;` - Returns 150-200+
- [ ] Multiple "Mathematics" entries exist with different codes
- [ ] All grades have subjects linked
- [ ] No duplicate code errors

---

## 🐛 **Troubleshooting**

### **Error: "Can't DROP 'name'; check that column/key exists"**

```sql
-- The constraint might have a different name. Try:
SHOW INDEXES FROM subjects;

-- Look for the constraint name and drop it:
ALTER TABLE subjects DROP INDEX [actual_constraint_name];
```

### **Error: "Duplicate entry for 'code'"**

```sql
-- Find duplicate codes:
SELECT code, COUNT(*) FROM subjects GROUP BY code HAVING COUNT(*) > 1;

-- Remove duplicates (keeps first occurrence):
DELETE s1 FROM subjects s1
INNER JOIN subjects s2 
WHERE s1.id > s2.id AND s1.code = s2.code;
```

### **Import Still Shows Warnings**

```
Warning: #1062 Duplicate entry...
```

**This is OK!** The script uses `INSERT IGNORE` which skips duplicates. As long as you see:
- "X rows inserted" for each query
- No actual ERROR (warnings are OK)
- The import completes successfully

Then it worked!

---

## 🎓 **Why This Design is Better**

### **OLD (WRONG) Design:**
```
❌ Only one "Mathematics" in entire database
❌ Can't have same subject for different grades
❌ Forces ugly names like "Mathematics Grade 4"
❌ Doesn't match real curriculum structure
❌ Breaks Moodle mapping (need separate courses per grade)
```

### **NEW (CORRECT) Design:**
```
✅ Same subject name can exist multiple times
✅ Differentiated by unique code (G4-MAT-003, G5-MAT-003)
✅ Clean, readable subject names
✅ Matches real-world curriculum
✅ Perfect for Moodle (map each grade to different course)
✅ Scalable for multi-tenancy
```

---

## 📊 **Before vs After**

### **Before Fix:**

```sql
SELECT * FROM subjects WHERE name = 'Mathematics';
```

Result:
```
id | name        | code
1  | Mathematics | MAT7100
```

Only 1 Mathematics!

### **After Fix:**

```sql
SELECT * FROM subjects WHERE name LIKE '%Mathematics%' ORDER BY code;
```

Result:
```
id  | name                          | code
1   | Mathematics                   | G4-MAT-003
2   | Mathematics                   | G5-MAT-003
3   | Mathematics                   | G6-MAT-003
15  | Mathematics (Arithmetic...)   | G7-MAT-003
28  | Mathematics                   | F1-MAT-003
45  | Mathematics (Algebra...)      | F2-MAT-003
...and more
```

Multiple Mathematics entries! ✅

---

## 🔄 **Rollback (If Needed)**

If something goes wrong:

```sql
-- Restore from backup
DROP TABLE subjects;
CREATE TABLE subjects AS SELECT * FROM subjects_backup;

DROP TABLE grade_subject;
CREATE TABLE grade_subject AS SELECT * FROM grade_subject_backup;
```

---

## 🎉 **Success Indicators**

You'll know it worked when:

1. ✅ **No errors** during table structure change
2. ✅ **No errors** during curriculum import
3. ✅ `SELECT COUNT(*) FROM subjects` returns **150-200+**
4. ✅ Multiple "Mathematics" entries with different codes
5. ✅ All grades show subjects in admin panel
6. ✅ Can map subjects to Moodle courses (each grade separately)

---

## 📞 **Next Steps After Fix**

1. ✅ **Verify import** (run verification queries)
2. ✅ **Test admin panel** (check "Configure Subjects")
3. ✅ **Import Moodle tables** (`ADD_MOODLE_INTEGRATION.sql`)
4. ✅ **Configure Moodle** (Settings → Moodle Integration)
5. ✅ **Map subjects** (Admin → Map Subjects to Moodle)
6. ✅ **Test payment** (Parent portal → Make Payment)
7. ✅ **Verify enrollment** (Check Moodle course participants)

---

## 🚀 **Summary**

**Problem:** UNIQUE constraint on subject name prevented curriculum import  
**Solution:** Remove UNIQUE from name, ensure UNIQUE on code  
**Time:** 10 minutes total  
**Risk:** Low (includes backups and rollback)  
**Result:** Complete Eswatini curriculum with Moodle integration ready!  

---

**Run the fix now and your curriculum import will work perfectly! 🎯**

---

*File: FIX_SUBJECTS_DUPLICATE_GUIDE.md*  
*Status: ✅ Complete Step-by-Step Guide*  
*Last Updated: 2025*

