-- =====================================================
-- SKOLO-KINE COMPLETE DATABASE SCHEMA
-- =====================================================
-- This file contains the complete database schema with all systems
-- including core tables, payment system, email system, monitoring,
-- security, wallet system, and all related functionality.
-- 
-- Generated: October 2025
-- Version: Production Ready - Complete System
-- =====================================================

SET SQL_MODE = "NO_AUTO_VALUE_ON_ZERO";
START TRANSACTION;
SET time_zone = "+00:00";

/*!40101 SET @OLD_CHARACTER_SET_CLIENT=@@CHARACTER_SET_CLIENT */;
/*!40101 SET @OLD_CHARACTER_SET_RESULTS=@@CHARACTER_SET_RESULTS */;
/*!40101 SET @OLD_COLLATION_CONNECTION=@@COLLATION_CONNECTION */;
/*!40101 SET NAMES utf8mb4 */;

-- =====================================================
-- CORE SYSTEM TABLES
-- =====================================================

-- Admins table
CREATE TABLE IF NOT EXISTS `admins` (
  `id` int(11) NOT NULL AUTO_INCREMENT,
  `username` varchar(64) NOT NULL,
  `password_hash` varchar(255) NOT NULL,
  `momo_number` varchar(32) DEFAULT NULL,
  `bank_account` varchar(50) DEFAULT NULL,
  `account_verified` tinyint(1) DEFAULT 0,
  `academy_reference` varchar(50) DEFAULT NULL,
  `created_at` timestamp NULL DEFAULT current_timestamp(),
  PRIMARY KEY (`id`),
  UNIQUE KEY `username` (`username`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci;

-- Grades table
CREATE TABLE IF NOT EXISTS `grades` (
  `id` int(11) NOT NULL AUTO_INCREMENT,
  `name` varchar(32) NOT NULL,
  PRIMARY KEY (`id`),
  UNIQUE KEY `name` (`name`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci;

-- Subjects table
CREATE TABLE IF NOT EXISTS `subjects` (
  `id` int(11) NOT NULL AUTO_INCREMENT,
  `name` varchar(128) NOT NULL,
  `code` varchar(32) NOT NULL,
  PRIMARY KEY (`id`),
  UNIQUE KEY `unique_code` (`code`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci;

-- Grade-Subject relationship table
CREATE TABLE IF NOT EXISTS `grade_subject` (
  `grade_id` int(11) NOT NULL,
  `subject_id` int(11) NOT NULL,
  PRIMARY KEY (`grade_id`,`subject_id`),
  KEY `idx_grade_subject_subject_id` (`subject_id`),
  CONSTRAINT `fk_grade_subject_grade` FOREIGN KEY (`grade_id`) REFERENCES `grades` (`id`) ON DELETE CASCADE ON UPDATE CASCADE,
  CONSTRAINT `fk_grade_subject_subject` FOREIGN KEY (`subject_id`) REFERENCES `subjects` (`id`) ON DELETE CASCADE ON UPDATE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci;

-- Students table (Enhanced with payment, soft deletion, academy reference, and Moodle integration)
CREATE TABLE IF NOT EXISTS `students` (
  `id` int(11) NOT NULL AUTO_INCREMENT,
  `first_name` varchar(100) NOT NULL,
  `last_name` varchar(100) NOT NULL,
  `email` varchar(191) NOT NULL,
  `grade_id` int(11) NOT NULL,
  `username` varchar(64) NOT NULL,
  `password_hash` varchar(255) NOT NULL,
  `email_verified` tinyint(1) DEFAULT 0,
  `email_verification_token` varchar(255) DEFAULT NULL,
  `email_verification_expires` timestamp NULL DEFAULT NULL,
  `created_at` timestamp NULL DEFAULT current_timestamp(),
  `temp_password` varchar(255) DEFAULT NULL,
  `deleted_at` datetime DEFAULT NULL,
  `permanent_delete_date` datetime DEFAULT NULL,
  `status` enum('active','suspended') DEFAULT 'active',
  `payment_status` varchar(32) DEFAULT 'UNPAID',
  `payment_date` timestamp NULL DEFAULT NULL,
  `phone` varchar(32) DEFAULT NULL,
  `academy_reference` varchar(100) DEFAULT NULL,
  `last_payment_date` datetime DEFAULT NULL,
  `next_payment_due` datetime DEFAULT NULL,
  `payment_cycle_start` datetime DEFAULT NULL,
  `payment_reminder_sent` tinyint(1) DEFAULT 0,
  `warning_email_sent` tinyint(1) DEFAULT 0,
  `suspension_warning_sent` tinyint(1) DEFAULT 0,
  `moodle_status` varchar(20) DEFAULT 'active',
  `suspended_date` datetime DEFAULT NULL,
  PRIMARY KEY (`id`),
  UNIQUE KEY `email` (`email`),
  UNIQUE KEY `username` (`username`),
  KEY `idx_students_grade_id` (`grade_id`),
  KEY `idx_students_email_verified` (`email_verified`),
  KEY `idx_students_verification_token` (`email_verification_token`),
  KEY `idx_students_status` (`status`),
  KEY `idx_students_deleted_at` (`deleted_at`),
  KEY `idx_students_permanent_delete_date` (`permanent_delete_date`),
  KEY `idx_students_payment_status` (`payment_status`),
  KEY `idx_students_payment_date` (`payment_date`),
  KEY `idx_students_academy_ref` (`academy_reference`),
  KEY `idx_students_moodle_status` (`moodle_status`),
  KEY `idx_students_next_payment_due` (`next_payment_due`),
  CONSTRAINT `fk_students_grade` FOREIGN KEY (`grade_id`) REFERENCES `grades` (`id`) ON UPDATE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci;

-- Student-Subject relationship table
CREATE TABLE IF NOT EXISTS `student_subject` (
  `student_id` int(11) NOT NULL,
  `subject_id` int(11) NOT NULL,
  PRIMARY KEY (`student_id`,`subject_id`),
  KEY `idx_student_subject_subject_id` (`subject_id`),
  CONSTRAINT `fk_student_subject_student` FOREIGN KEY (`student_id`) REFERENCES `students` (`id`) ON DELETE CASCADE ON UPDATE CASCADE,
  CONSTRAINT `fk_student_subject_subject` FOREIGN KEY (`subject_id`) REFERENCES `subjects` (`id`) ON DELETE CASCADE ON UPDATE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci;

-- =====================================================
-- PAYMENT SYSTEM TABLES
-- =====================================================

-- Payments table for MTN MoMo and Card integration
CREATE TABLE IF NOT EXISTS `payments` (
    `id` int(11) NOT NULL AUTO_INCREMENT,
    `student_id` int(11) NOT NULL,
    `amount` decimal(10,2) NOT NULL,
    `currency` varchar(8) NOT NULL DEFAULT 'SZL',
    `reference_id` varchar(64) NOT NULL,
    `external_id` varchar(64) NOT NULL,
    `msisdn` varchar(32) NOT NULL,
    `status` varchar(32) NOT NULL DEFAULT 'PENDING',
    `payment_method` varchar(20) DEFAULT 'MOMO',
    `momo_status` varchar(50) DEFAULT NULL,
    `raw_response` text DEFAULT NULL,
    `error_message` text DEFAULT NULL,
    `academy_reference` varchar(100) DEFAULT NULL,
    `platform_fee` decimal(10,2) DEFAULT 0.50,
    `admin_earnings` decimal(10,2) DEFAULT NULL,
    `created_at` timestamp DEFAULT CURRENT_TIMESTAMP,
    `updated_at` timestamp DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    PRIMARY KEY (`id`),
    UNIQUE KEY `reference_id` (`reference_id`),
    KEY `idx_payments_student_id` (`student_id`),
    KEY `idx_payments_external_id` (`external_id`),
    KEY `idx_payments_status` (`status`),
    KEY `idx_payments_payment_method` (`payment_method`),
    KEY `idx_payments_created_at` (`created_at`),
    KEY `idx_payments_academy_ref` (`academy_reference`),
    CONSTRAINT `fk_payments_student` FOREIGN KEY (`student_id`) REFERENCES `students` (`id`) ON DELETE CASCADE ON UPDATE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci;

-- Payment items table (for detailed payment tracking)
CREATE TABLE IF NOT EXISTS `payment_items` (
    `id` int(11) NOT NULL AUTO_INCREMENT,
    `payment_id` int(11) NOT NULL,
    `subject_id` int(11) NOT NULL,
    `price` decimal(10,2) NOT NULL DEFAULT 0.00,
    `created_at` timestamp DEFAULT CURRENT_TIMESTAMP,
    PRIMARY KEY (`id`),
    KEY `idx_payment_items_payment_id` (`payment_id`),
    KEY `idx_payment_items_subject_id` (`subject_id`),
    CONSTRAINT `fk_payment_items_payment` FOREIGN KEY (`payment_id`) REFERENCES `payments` (`id`) ON DELETE CASCADE,
    CONSTRAINT `fk_payment_items_subject` FOREIGN KEY (`subject_id`) REFERENCES `subjects` (`id`) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci;

-- Payment cycles table (for tracking monthly payment cycles)
CREATE TABLE IF NOT EXISTS `payment_cycles` (
    `id` int(11) NOT NULL AUTO_INCREMENT,
    `student_id` int(11) NOT NULL,
    `cycle_start` datetime NOT NULL,
    `cycle_end` datetime NOT NULL,
    `amount_due` decimal(10,2) NOT NULL,
    `amount_paid` decimal(10,2) DEFAULT 0.00,
    `status` enum('active','paid','overdue','suspended') DEFAULT 'active',
    `payment_date` datetime DEFAULT NULL,
    `created_at` timestamp DEFAULT CURRENT_TIMESTAMP,
    PRIMARY KEY (`id`),
    KEY `idx_student_cycle` (`student_id`,`cycle_start`),
    KEY `idx_status` (`status`),
    KEY `idx_cycle_end` (`cycle_end`),
    CONSTRAINT `fk_payment_cycles_student` FOREIGN KEY (`student_id`) REFERENCES `students` (`id`) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci;

-- =====================================================
-- EMAIL SYSTEM TABLES
-- =====================================================

-- Email queue table
CREATE TABLE IF NOT EXISTS `email_queue` (
    `id` int(11) NOT NULL AUTO_INCREMENT,
    `to_email` varchar(255) NOT NULL,
    `to_name` varchar(255) NOT NULL,
    `subject` varchar(500) NOT NULL,
    `html_content` longtext NOT NULL,
    `text_content` text DEFAULT NULL,
    `priority` enum('LOW','NORMAL','HIGH','URGENT') DEFAULT 'NORMAL',
    `status` enum('PENDING','SENDING','SENT','FAILED') DEFAULT 'PENDING',
    `attempts` int(11) DEFAULT 0,
    `max_attempts` int(11) DEFAULT 3,
    `error_message` text DEFAULT NULL,
    `created_at` timestamp DEFAULT CURRENT_TIMESTAMP,
    `processed_at` timestamp NULL DEFAULT NULL,
    `sent_at` timestamp NULL DEFAULT NULL,
    PRIMARY KEY (`id`),
    KEY `idx_status` (`status`),
    KEY `idx_priority` (`priority`),
    KEY `idx_created_at` (`created_at`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci;

-- Email logs table
CREATE TABLE IF NOT EXISTS `email_logs` (
    `id` int(11) NOT NULL AUTO_INCREMENT,
    `to_email` varchar(255) NOT NULL,
    `to_name` varchar(255) DEFAULT NULL,
    `subject` varchar(500) NOT NULL,
    `status` enum('SUCCESS','FAILED') NOT NULL,
    `error_message` text DEFAULT NULL,
    `sent_at` timestamp DEFAULT CURRENT_TIMESTAMP,
    PRIMARY KEY (`id`),
    KEY `idx_to_email` (`to_email`),
    KEY `idx_status` (`status`),
    KEY `idx_sent_at` (`sent_at`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci;

-- Email templates table
CREATE TABLE IF NOT EXISTS `email_templates` (
    `id` int(11) NOT NULL AUTO_INCREMENT,
    `name` varchar(100) NOT NULL,
    `subject` varchar(500) NOT NULL,
    `html_content` longtext NOT NULL,
    `text_content` text DEFAULT NULL,
    `variables` json DEFAULT NULL,
    `is_active` tinyint(1) DEFAULT 1,
    `created_at` timestamp DEFAULT CURRENT_TIMESTAMP,
    `updated_at` timestamp DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    PRIMARY KEY (`id`),
    UNIQUE KEY `name` (`name`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci;

-- =====================================================
-- SECURITY SYSTEM TABLES
-- =====================================================

-- Security logs table
CREATE TABLE IF NOT EXISTS `security_logs` (
    `id` int(11) NOT NULL AUTO_INCREMENT,
    `event` varchar(100) NOT NULL,
    `details` text DEFAULT NULL,
    `severity` enum('INFO','WARNING','ERROR','CRITICAL') DEFAULT 'INFO',
    `ip_address` varchar(45) NOT NULL,
    `user_agent` text DEFAULT NULL,
    `user_id` int(11) DEFAULT NULL,
    `created_at` timestamp DEFAULT CURRENT_TIMESTAMP,
    PRIMARY KEY (`id`),
    KEY `idx_event` (`event`),
    KEY `idx_severity` (`severity`),
    KEY `idx_ip_address` (`ip_address`),
    KEY `idx_created_at` (`created_at`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci;

-- Failed login attempts table
CREATE TABLE IF NOT EXISTS `failed_login_attempts` (
    `id` int(11) NOT NULL AUTO_INCREMENT,
    `username` varchar(255) NOT NULL,
    `ip_address` varchar(45) NOT NULL,
    `user_agent` text DEFAULT NULL,
    `attempted_at` timestamp DEFAULT CURRENT_TIMESTAMP,
    PRIMARY KEY (`id`),
    KEY `idx_username` (`username`),
    KEY `idx_ip_address` (`ip_address`),
    KEY `idx_attempted_at` (`attempted_at`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci;

-- User sessions table
CREATE TABLE IF NOT EXISTS `user_sessions` (
    `id` int(11) NOT NULL AUTO_INCREMENT,
    `user_id` int(11) NOT NULL,
    `session_id` varchar(128) NOT NULL,
    `ip_address` varchar(45) NOT NULL,
    `user_agent` text DEFAULT NULL,
    `created_at` timestamp DEFAULT CURRENT_TIMESTAMP,
    `last_activity` timestamp DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    `expires_at` timestamp NOT NULL DEFAULT (CURRENT_TIMESTAMP + INTERVAL 24 HOUR),
    `is_active` tinyint(1) DEFAULT 1,
    PRIMARY KEY (`id`),
    UNIQUE KEY `session_id` (`session_id`),
    KEY `idx_user_id` (`user_id`),
    KEY `idx_expires_at` (`expires_at`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci;

-- =====================================================
-- MONITORING SYSTEM TABLES
-- =====================================================

-- System performance monitoring
CREATE TABLE IF NOT EXISTS `system_metrics` (
    `id` int(11) NOT NULL AUTO_INCREMENT,
    `metric_name` varchar(100) NOT NULL,
    `metric_value` decimal(10,4) NOT NULL,
    `metric_unit` varchar(20) NOT NULL,
    `server_info` json DEFAULT NULL,
    `created_at` timestamp DEFAULT CURRENT_TIMESTAMP,
    PRIMARY KEY (`id`),
    KEY `idx_metric_name` (`metric_name`),
    KEY `idx_created_at` (`created_at`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci;

-- Application performance monitoring
CREATE TABLE IF NOT EXISTS `app_performance` (
    `id` int(11) NOT NULL AUTO_INCREMENT,
    `page_name` varchar(100) NOT NULL,
    `response_time` decimal(8,3) NOT NULL,
    `memory_usage` int(11) NOT NULL,
    `query_count` int(11) DEFAULT 0,
    `error_count` int(11) DEFAULT 0,
    `user_agent` text DEFAULT NULL,
    `ip_address` varchar(45) DEFAULT NULL,
    `created_at` timestamp DEFAULT CURRENT_TIMESTAMP,
    PRIMARY KEY (`id`),
    KEY `idx_page_name` (`page_name`),
    KEY `idx_created_at` (`created_at`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci;

-- Database performance monitoring
CREATE TABLE IF NOT EXISTS `database_metrics` (
    `id` int(11) NOT NULL AUTO_INCREMENT,
    `query_type` varchar(50) NOT NULL,
    `execution_time` decimal(8,3) NOT NULL,
    `rows_affected` int(11) DEFAULT 0,
    `query_text` text DEFAULT NULL,
    `created_at` timestamp DEFAULT CURRENT_TIMESTAMP,
    PRIMARY KEY (`id`),
    KEY `idx_query_type` (`query_type`),
    KEY `idx_created_at` (`created_at`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci;

-- User activity monitoring
CREATE TABLE IF NOT EXISTS `user_activity` (
    `id` int(11) NOT NULL AUTO_INCREMENT,
    `user_id` int(11) DEFAULT NULL,
    `user_type` enum('STUDENT','ADMIN','GUEST') NOT NULL,
    `action` varchar(100) NOT NULL,
    `page_url` varchar(500) DEFAULT NULL,
    `ip_address` varchar(45) NOT NULL,
    `user_agent` text DEFAULT NULL,
    `session_id` varchar(128) DEFAULT NULL,
    `created_at` timestamp DEFAULT CURRENT_TIMESTAMP,
    PRIMARY KEY (`id`),
    KEY `idx_user_id` (`user_id`),
    KEY `idx_user_type` (`user_type`),
    KEY `idx_action` (`action`),
    KEY `idx_created_at` (`created_at`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci;

-- Error logging
CREATE TABLE IF NOT EXISTS `error_logs` (
    `id` int(11) NOT NULL AUTO_INCREMENT,
    `error_type` varchar(50) NOT NULL,
    `error_message` text NOT NULL,
    `error_file` varchar(500) DEFAULT NULL,
    `error_line` int(11) DEFAULT NULL,
    `stack_trace` text DEFAULT NULL,
    `user_id` int(11) DEFAULT NULL,
    `ip_address` varchar(45) DEFAULT NULL,
    `user_agent` text DEFAULT NULL,
    `created_at` timestamp DEFAULT CURRENT_TIMESTAMP,
    PRIMARY KEY (`id`),
    KEY `idx_error_type` (`error_type`),
    KEY `idx_created_at` (`created_at`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci;

-- API monitoring
CREATE TABLE IF NOT EXISTS `api_monitoring` (
    `id` int(11) NOT NULL AUTO_INCREMENT,
    `api_endpoint` varchar(200) NOT NULL,
    `method` varchar(10) NOT NULL,
    `response_code` int(11) NOT NULL,
    `response_time` decimal(8,3) NOT NULL,
    `request_size` int(11) DEFAULT 0,
    `response_size` int(11) DEFAULT 0,
    `user_id` int(11) DEFAULT NULL,
    `ip_address` varchar(45) DEFAULT NULL,
    `created_at` timestamp DEFAULT CURRENT_TIMESTAMP,
    PRIMARY KEY (`id`),
    KEY `idx_api_endpoint` (`api_endpoint`),
    KEY `idx_response_code` (`response_code`),
    KEY `idx_created_at` (`created_at`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci;

-- Bot monitoring
CREATE TABLE IF NOT EXISTS `bot_monitoring` (
    `id` int(11) NOT NULL AUTO_INCREMENT,
    `bot_name` varchar(100) NOT NULL,
    `status` enum('RUNNING','STOPPED','ERROR','IDLE') NOT NULL,
    `last_run` timestamp NULL DEFAULT NULL,
    `next_run` timestamp NULL DEFAULT NULL,
    `execution_time` decimal(8,3) DEFAULT NULL,
    `records_processed` int(11) DEFAULT 0,
    `errors_count` int(11) DEFAULT 0,
    `memory_usage` int(11) DEFAULT NULL,
    `cpu_usage` decimal(5,2) DEFAULT NULL,
    `created_at` timestamp DEFAULT CURRENT_TIMESTAMP,
    `updated_at` timestamp DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    PRIMARY KEY (`id`),
    KEY `idx_bot_name` (`bot_name`),
    KEY `idx_status` (`status`),
    KEY `idx_last_run` (`last_run`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci;

-- System health checks
CREATE TABLE IF NOT EXISTS `health_checks` (
    `id` int(11) NOT NULL AUTO_INCREMENT,
    `check_name` varchar(100) NOT NULL,
    `status` enum('HEALTHY','WARNING','CRITICAL','UNKNOWN') NOT NULL,
    `check_value` decimal(10,4) DEFAULT NULL,
    `threshold_warning` decimal(10,4) DEFAULT NULL,
    `threshold_critical` decimal(10,4) DEFAULT NULL,
    `message` text DEFAULT NULL,
    `created_at` timestamp DEFAULT CURRENT_TIMESTAMP,
    PRIMARY KEY (`id`),
    KEY `idx_check_name` (`check_name`),
    KEY `idx_status` (`status`),
    KEY `idx_created_at` (`created_at`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci;

-- Alert system
CREATE TABLE IF NOT EXISTS `alerts` (
    `id` int(11) NOT NULL AUTO_INCREMENT,
    `alert_type` varchar(50) NOT NULL,
    `severity` enum('LOW','MEDIUM','HIGH','CRITICAL') NOT NULL,
    `title` varchar(200) NOT NULL,
    `message` text NOT NULL,
    `status` enum('ACTIVE','ACKNOWLEDGED','RESOLVED','SUPPRESSED') DEFAULT 'ACTIVE',
    `acknowledged_by` int(11) DEFAULT NULL,
    `acknowledged_at` timestamp NULL DEFAULT NULL,
    `resolved_at` timestamp NULL DEFAULT NULL,
    `created_at` timestamp DEFAULT CURRENT_TIMESTAMP,
    PRIMARY KEY (`id`),
    KEY `idx_alert_type` (`alert_type`),
    KEY `idx_severity` (`severity`),
    KEY `idx_status` (`status`),
    KEY `idx_created_at` (`created_at`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci;

-- System uptime tracking
CREATE TABLE IF NOT EXISTS `uptime_tracking` (
    `id` int(11) NOT NULL AUTO_INCREMENT,
    `service_name` varchar(100) NOT NULL,
    `status` enum('UP','DOWN','DEGRADED') NOT NULL,
    `response_time` decimal(8,3) DEFAULT NULL,
    `error_message` text DEFAULT NULL,
    `checked_at` timestamp DEFAULT CURRENT_TIMESTAMP,
    PRIMARY KEY (`id`),
    KEY `idx_service_name` (`service_name`),
    KEY `idx_status` (`status`),
    KEY `idx_checked_at` (`checked_at`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci;

-- =====================================================
-- WALLET SYSTEM TABLES
-- =====================================================

-- Admin wallet to track balances
CREATE TABLE IF NOT EXISTS `admin_wallet` (
    `id` int(11) NOT NULL AUTO_INCREMENT,
    `admin_id` int(11) NOT NULL,
    `balance` decimal(10,2) NOT NULL DEFAULT 0.00,
    `total_earned` decimal(10,2) NOT NULL DEFAULT 0.00,
    `total_withdrawn` decimal(10,2) NOT NULL DEFAULT 0.00,
    `pending_withdrawal` decimal(10,2) NOT NULL DEFAULT 0.00,
    `currency` varchar(8) NOT NULL DEFAULT 'SZL',
    `momo_number` varchar(32) DEFAULT NULL,
    `account_verified` tinyint(1) DEFAULT 0,
    `created_at` timestamp DEFAULT CURRENT_TIMESTAMP,
    `updated_at` timestamp DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    PRIMARY KEY (`id`),
    UNIQUE KEY `admin_id` (`admin_id`),
    CONSTRAINT `fk_admin_wallet` FOREIGN KEY (`admin_id`) REFERENCES `admins`(`id`) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci;

-- Wallet transactions (deposits from payments)
CREATE TABLE IF NOT EXISTS `wallet_transactions` (
    `id` int(11) NOT NULL AUTO_INCREMENT,
    `admin_id` int(11) NOT NULL,
    `payment_id` int(11) DEFAULT NULL,
    `withdrawal_id` int(11) DEFAULT NULL,
    `type` enum('CREDIT','DEBIT','WITHDRAWAL','REFUND','PLATFORM_FEE') NOT NULL,
    `amount` decimal(10,2) NOT NULL,
    `balance_before` decimal(10,2) NOT NULL,
    `balance_after` decimal(10,2) NOT NULL,
    `description` varchar(255) DEFAULT NULL,
    `reference` varchar(100) DEFAULT NULL,
    `created_at` timestamp DEFAULT CURRENT_TIMESTAMP,
    PRIMARY KEY (`id`),
    KEY `idx_admin_id` (`admin_id`),
    KEY `idx_payment_id` (`payment_id`),
    KEY `idx_withdrawal_id` (`withdrawal_id`),
    KEY `idx_type` (`type`),
    KEY `idx_created_at` (`created_at`),
    CONSTRAINT `fk_wallet_txn_admin` FOREIGN KEY (`admin_id`) REFERENCES `admins`(`id`) ON DELETE CASCADE,
    CONSTRAINT `fk_wallet_txn_payment` FOREIGN KEY (`payment_id`) REFERENCES `payments`(`id`) ON DELETE SET NULL
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci;

-- Withdrawal requests
CREATE TABLE IF NOT EXISTS `withdrawal_requests` (
    `id` int(11) NOT NULL AUTO_INCREMENT,
    `admin_id` int(11) NOT NULL,
    `amount` decimal(10,2) NOT NULL,
    `momo_number` varchar(32) NOT NULL,
    `status` enum('PENDING','PROCESSING','COMPLETED','FAILED','CANCELLED') DEFAULT 'PENDING',
    `momo_reference_id` varchar(64) DEFAULT NULL,
    `momo_response` text DEFAULT NULL,
    `error_message` text DEFAULT NULL,
    `requested_at` timestamp DEFAULT CURRENT_TIMESTAMP,
    `processed_at` timestamp NULL DEFAULT NULL,
    `completed_at` timestamp NULL DEFAULT NULL,
    PRIMARY KEY (`id`),
    KEY `idx_admin_id` (`admin_id`),
    KEY `idx_status` (`status`),
    KEY `idx_momo_reference_id` (`momo_reference_id`),
    KEY `idx_requested_at` (`requested_at`),
    CONSTRAINT `fk_withdrawal_admin` FOREIGN KEY (`admin_id`) REFERENCES `admins`(`id`) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci;

-- Platform fees tracking (your earnings)
CREATE TABLE IF NOT EXISTS `platform_fees` (
    `id` int(11) NOT NULL AUTO_INCREMENT,
    `payment_id` int(11) NOT NULL,
    `admin_id` int(11) NOT NULL,
    `gross_amount` decimal(10,2) NOT NULL,
    `fee_amount` decimal(10,2) NOT NULL DEFAULT 0.50,
    `net_amount` decimal(10,2) NOT NULL,
    `payment_method` varchar(20) DEFAULT 'MOMO',
    `academy_reference` varchar(100) DEFAULT NULL,
    `created_at` timestamp DEFAULT CURRENT_TIMESTAMP,
    PRIMARY KEY (`id`),
    KEY `idx_payment_id` (`payment_id`),
    KEY `idx_admin_id` (`admin_id`),
    KEY `idx_payment_method` (`payment_method`),
    KEY `idx_academy_reference` (`academy_reference`),
    KEY `idx_created_at` (`created_at`),
    CONSTRAINT `fk_fee_payment` FOREIGN KEY (`payment_id`) REFERENCES `payments`(`id`) ON DELETE CASCADE,
    CONSTRAINT `fk_fee_admin` FOREIGN KEY (`admin_id`) REFERENCES `admins`(`id`) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci;

-- Academy/School references for tracking payment sources
CREATE TABLE IF NOT EXISTS `academy_references` (
    `id` int(11) NOT NULL AUTO_INCREMENT,
    `admin_id` int(11) NOT NULL,
    `academy_name` varchar(100) NOT NULL,
    `reference_code` varchar(50) NOT NULL,
    `description` text DEFAULT NULL,
    `is_active` tinyint(1) DEFAULT 1,
    `total_students` int(11) DEFAULT 0,
    `total_payments` decimal(10,2) DEFAULT 0.00,
    `created_at` timestamp DEFAULT CURRENT_TIMESTAMP,
    `updated_at` timestamp DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    PRIMARY KEY (`id`),
    UNIQUE KEY `reference_code` (`reference_code`),
    KEY `idx_admin_id` (`admin_id`),
    CONSTRAINT `fk_academy_admin` FOREIGN KEY (`admin_id`) REFERENCES `admins`(`id`) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci;

-- =====================================================
-- INSERT DEFAULT DATA
-- =====================================================

-- Insert default admin
INSERT INTO `admins` (`username`, `password_hash`, `academy_reference`) VALUES
('admin', '$2y$10$92IXUNpkjO0rOQ5byMi.Ye4oKoEa3Ro9llC/.og/at2.uheWG/igi', 'KINE');

-- Insert default grades
INSERT INTO `grades` (`id`, `name`) VALUES
(1, 'Grade 7'),
(2, 'Grade 4'),
(3, 'Grade 5'),
(4, 'Grade 6'),
(6, 'Form 1'),
(7, 'Form 2'),
(8, 'Form 3'),
(9, 'Form 4'),
(10, 'Form 5');

-- Insert default subjects
INSERT INTO `subjects` (`id`, `name`, `code`) VALUES
(1, 'Mathematics', 'MAT7100'),
(2, 'English', 'ENG7100'),
(3, 'SiSwati', 'SIS7100'),
(4, 'Religious Education', 'REL7100'),
(5, 'Social Studies', 'SOC7100'),
(6, 'Expressive Arts', 'EXP7100'),
(7, 'Agriculture', 'AGR7100'),
(8, 'Consumer Science', 'CON7100'),
(9, 'HPE', 'HPE7100'),
(10, 'Science', 'SCI7100'),
(11, 'ICT', 'ICT7100');

-- Insert default email templates
INSERT INTO `email_templates` (`name`, `subject`, `html_content`, `variables`) VALUES
('verification', 'Verify Your Email - {{app_name}}', '<!DOCTYPE html><html><head><meta charset="UTF-8"><title>Email Verification</title></head><body><h1>Verify Your Email</h1><p>Hello {{student_name}},</p><p>Please click the link below to verify your email address:</p><p><a href="{{verification_link}}">Verify Email</a></p><p>Best regards,<br>{{app_name}} Team</p></body></html>', '["app_name", "student_name", "verification_link"]'),
('welcome', 'Welcome to {{app_name}}!', '<!DOCTYPE html><html><head><meta charset="UTF-8"><title>Welcome</title></head><body><h1>Welcome to {{app_name}}!</h1><p>Hello {{student_name}},</p><p>Your account has been successfully created. You can now access your subjects:</p><ul>{{subjects}}</ul><p><a href="{{login_url}}">Login to your account</a></p><p>Best regards,<br>{{app_name}} Team</p></body></html>', '["app_name", "student_name", "login_url", "subjects"]'),
('payment_confirmation', 'Payment Confirmation - {{app_name}}', '<!DOCTYPE html><html><head><meta charset="UTF-8"><title>Payment Confirmation</title></head><body><h1>Payment Confirmed</h1><p>Hello {{student_name}},</p><p>Your payment of {{amount}} has been successfully processed.</p><p>Subjects enrolled:</p><ul>{{subjects}}</ul><p>Best regards,<br>{{app_name}} Team</p></body></html>', '["app_name", "student_name", "amount", "subjects"]'),
('password_reset', 'Password Reset - {{app_name}}', '<!DOCTYPE html><html><head><meta charset="UTF-8"><title>Password Reset</title></head><body><h1>Password Reset</h1><p>Hello {{student_name}},</p><p>Click the link below to reset your password:</p><p><a href="{{reset_link}}">Reset Password</a></p><p>Best regards,<br>{{app_name}} Team</p></body></html>', '["app_name", "student_name", "reset_link"]')
ON DUPLICATE KEY UPDATE `updated_at` = CURRENT_TIMESTAMP;

-- Insert default academy reference for Kine Academy
INSERT INTO `academy_references` (`admin_id`, `academy_name`, `reference_code`, `description`, `is_active`) VALUES
(1, 'Kine Academy', 'KINE', 'Main Kine Academy', 1);

-- Insert default platform admin wallet
INSERT INTO `admin_wallet` (`admin_id`, `balance`, `currency`) VALUES
(1, 0.00, 'SZL');

-- =====================================================
-- CREATE INDEXES FOR PERFORMANCE
-- =====================================================

-- Additional indexes for better performance
CREATE INDEX IF NOT EXISTS `idx_students_email_verified` ON `students` (`email_verified`);
CREATE INDEX IF NOT EXISTS `idx_students_verification_token` ON `students` (`email_verification_token`);
CREATE INDEX IF NOT EXISTS `idx_students_status` ON `students` (`status`);
CREATE INDEX IF NOT EXISTS `idx_students_deleted_at` ON `students` (`deleted_at`);
CREATE INDEX IF NOT EXISTS `idx_students_permanent_delete_date` ON `students` (`permanent_delete_date`);
CREATE INDEX IF NOT EXISTS `idx_students_payment_status` ON `students` (`payment_status`);
CREATE INDEX IF NOT EXISTS `idx_students_payment_date` ON `students` (`payment_date`);

-- =====================================================
-- CREATE VIEWS FOR REPORTING
-- =====================================================

-- Admin wallet summary view
CREATE OR REPLACE VIEW `v_admin_wallet_summary` AS
SELECT 
    aw.admin_id,
    a.username,
    aw.balance,
    aw.total_earned,
    aw.total_withdrawn,
    aw.pending_withdrawal,
    aw.momo_number,
    aw.account_verified,
    COUNT(DISTINCT p.id) as total_transactions,
    SUM(pf.fee_amount) as total_platform_fees,
    aw.updated_at
FROM admin_wallet aw
LEFT JOIN admins a ON aw.admin_id = a.id
LEFT JOIN payments p ON p.status = 'SUCCESSFUL'
LEFT JOIN platform_fees pf ON pf.admin_id = aw.admin_id
GROUP BY aw.admin_id;

-- Academy performance view
CREATE OR REPLACE VIEW `v_academy_performance` AS
SELECT 
    ar.id,
    ar.admin_id,
    ar.academy_name,
    ar.reference_code,
    COUNT(DISTINCT s.id) as total_students,
    COUNT(DISTINCT p.id) as total_payments,
    SUM(p.amount) as total_revenue,
    SUM(pf.fee_amount) as total_platform_fees,
    SUM(pf.net_amount) as total_admin_earnings,
    ar.created_at
FROM academy_references ar
LEFT JOIN students s ON s.academy_reference = ar.reference_code
LEFT JOIN payments p ON p.academy_reference = ar.reference_code AND p.status = 'SUCCESSFUL'
LEFT JOIN platform_fees pf ON pf.academy_reference = ar.reference_code
WHERE ar.is_active = 1
GROUP BY ar.id;

-- =====================================================
-- CREATE STORED PROCEDURES
-- =====================================================

DELIMITER //

-- Procedure to credit admin wallet when payment is successful
CREATE PROCEDURE IF NOT EXISTS `sp_credit_admin_wallet`(
    IN p_payment_id INT,
    IN p_admin_id INT,
    IN p_academy_reference VARCHAR(100)
)
BEGIN
    DECLARE v_amount DECIMAL(10,2);
    DECLARE v_platform_fee DECIMAL(10,2) DEFAULT 0.50;
    DECLARE v_net_amount DECIMAL(10,2);
    DECLARE v_balance_before DECIMAL(10,2);
    DECLARE v_balance_after DECIMAL(10,2);
    
    -- Get payment amount
    SELECT amount INTO v_amount FROM payments WHERE id = p_payment_id;
    
    -- Calculate net amount (payment - platform fee)
    SET v_net_amount = v_amount - v_platform_fee;
    
    -- Get current balance
    SELECT COALESCE(balance, 0) INTO v_balance_before 
    FROM admin_wallet WHERE admin_id = p_admin_id;
    
    SET v_balance_after = v_balance_before + v_net_amount;
    
    -- Update wallet
    INSERT INTO admin_wallet (admin_id, balance, total_earned, currency)
    VALUES (p_admin_id, v_net_amount, v_net_amount, 'SZL')
    ON DUPLICATE KEY UPDATE
        balance = balance + v_net_amount,
        total_earned = total_earned + v_net_amount,
        updated_at = NOW();
    
    -- Record platform fee
    INSERT INTO platform_fees (payment_id, admin_id, gross_amount, fee_amount, net_amount, academy_reference)
    VALUES (p_payment_id, p_admin_id, v_amount, v_platform_fee, v_net_amount, p_academy_reference);
    
    -- Record wallet transaction
    INSERT INTO wallet_transactions 
    (admin_id, payment_id, type, amount, balance_before, balance_after, description, reference)
    VALUES 
    (p_admin_id, p_payment_id, 'CREDIT', v_net_amount, v_balance_before, v_balance_after, 
     CONCAT('Payment from student (', p_academy_reference, ')'), p_academy_reference);
    
    -- Update payment record
    UPDATE payments 
    SET platform_fee = v_platform_fee,
        admin_earnings = v_net_amount,
        updated_at = NOW()
    WHERE id = p_payment_id;
    
    -- Update academy reference stats
    UPDATE academy_references 
    SET total_students = total_students + 1,
        total_payments = total_payments + v_amount,
        updated_at = NOW()
    WHERE reference_code = p_academy_reference;
END //

-- Procedure to process withdrawal refund if failed
CREATE PROCEDURE IF NOT EXISTS `sp_refund_withdrawal`(
    IN p_withdrawal_id INT
)
BEGIN
    DECLARE v_admin_id INT;
    DECLARE v_amount DECIMAL(10,2);
    DECLARE v_balance_before DECIMAL(10,2);
    DECLARE v_balance_after DECIMAL(10,2);
    
    -- Get withdrawal details
    SELECT admin_id, amount INTO v_admin_id, v_amount
    FROM withdrawal_requests WHERE id = p_withdrawal_id;
    
    -- Get current balance
    SELECT balance INTO v_balance_before 
    FROM admin_wallet WHERE admin_id = v_admin_id;
    
    SET v_balance_after = v_balance_before + v_amount;
    
    -- Refund to wallet
    UPDATE admin_wallet 
    SET balance = balance + v_amount,
        total_withdrawn = total_withdrawn - v_amount,
        pending_withdrawal = pending_withdrawal - v_amount,
        updated_at = NOW()
    WHERE admin_id = v_admin_id;
    
    -- Record refund transaction
    INSERT INTO wallet_transactions 
    (admin_id, withdrawal_id, type, amount, balance_before, balance_after, description)
    VALUES 
    (v_admin_id, p_withdrawal_id, 'REFUND', v_amount, v_balance_before, v_balance_after, 
     'Withdrawal refund - transfer failed');
END //

DELIMITER ;

-- =====================================================
-- SET AUTO_INCREMENT VALUES
-- =====================================================

ALTER TABLE `admins` AUTO_INCREMENT = 1;
ALTER TABLE `grades` AUTO_INCREMENT = 65;
ALTER TABLE `students` AUTO_INCREMENT = 526;
ALTER TABLE `subjects` AUTO_INCREMENT = 12;

-- =====================================================
-- COMMIT TRANSACTION
-- =====================================================

COMMIT;

/*!40101 SET CHARACTER_SET_CLIENT=@OLD_CHARACTER_SET_CLIENT */;
/*!40101 SET CHARACTER_SET_RESULTS=@OLD_CHARACTER_SET_RESULTS */;
/*!40101 SET COLLATION_CONNECTION=@OLD_COLLATION_CONNECTION */;

-- =====================================================
-- END OF COMPLETE DATABASE SCHEMA
-- =====================================================
