<?php
/**
 * Tenant Directory Manager - Usage Examples
 * 
 * Demonstrates how to use the TenantDirectoryManager class
 */

require_once 'config.php';
require_once 'includes/tenant_directory_manager.php';

// Create database connection
$conn = new PDO(
    "mysql:host=" . DB_HOST . ";dbname=" . DB_NAME,
    DB_USER,
    DB_PASS
);

// Initialize manager
$manager = new TenantDirectoryManager($conn);

// =====================================================
// EXAMPLE 1: Initialize New Tenant
// =====================================================

echo "=== Example 1: Initialize New Tenant ===\n\n";

$tenant_id = 'soshigh_demo';
$tenant_name = 'SOS SOSHIGH Secondary School';

$result = $manager->initializeTenant($tenant_id, $tenant_name);

if ($result['success']) {
    echo "✅ Tenant initialized successfully!\n";
    echo "Path: {$result['path']}\n";
    echo "Directories created: {$result['directories_created']}\n";
    print_r($result['verification']);
} else {
    echo "❌ Error: {$result['error']}\n";
}

// =====================================================
// EXAMPLE 2: Initialize with Custom Settings
// =====================================================

echo "\n=== Example 2: Initialize with Custom Settings ===\n\n";

$custom_settings = [
    'storage_quota' => 5368709120, // 5GB
    'file_retention_days' => 730,   // 2 years
    'temp_cleanup_days' => 3,
    'backup_retention_days' => 90,
    'enable_auto_backup' => true,
    'max_file_size' => 52428800    // 50MB
];

$result = $manager->initializeTenant('premium_school', 'Premium Academy', $custom_settings);

if ($result['success']) {
    echo "✅ Premium tenant initialized with custom quota: 5GB\n";
}

// =====================================================
// EXAMPLE 3: Get Tenant Paths
// =====================================================

echo "\n=== Example 3: Get Tenant Paths ===\n\n";

// Root path
$rootPath = $manager->getTenantRootPath('soshigh_demo');
echo "Root path: {$rootPath}\n";

// Upload paths
$documentsPath = $manager->getTenantUploadPath('soshigh_demo', 'documents');
echo "Documents: {$documentsPath}\n";

$photosPath = $manager->getTenantUploadPath('soshigh_demo', 'profile_photos');
echo "Photos: {$photosPath}\n";

$receiptsPath = $manager->getTenantUploadPath('soshigh_demo', 'payment_receipts');
echo "Receipts: {$receiptsPath}\n";

// Backup paths
$backupPath = $manager->getTenantBackupPath('soshigh_demo');
echo "Backups: {$backupPath}\n";

$dbBackupPath = $manager->getTenantBackupPath('soshigh_demo', 'database');
echo "DB Backups: {$dbBackupPath}\n";

// Log paths
$logPath = $manager->getTenantLogPath('soshigh_demo');
echo "Logs: {$logPath}\n";

$errorLogPath = $manager->getTenantLogPath('soshigh_demo', 'errors');
echo "Error logs: {$errorLogPath}\n";

// =====================================================
// EXAMPLE 4: Verify Tenant Directories
// =====================================================

echo "\n=== Example 4: Verify Tenant Directories ===\n\n";

$verification = $manager->verifyTenantDirectories('soshigh_demo');

if ($verification['success']) {
    echo "✅ All directories verified!\n";
    echo "Directories OK: {$verification['directories_ok']}\n";
    echo "Writable: {$verification['directories_writable']}\n";
} else {
    echo "⚠️ Verification issues:\n";
    echo "Missing: {$verification['directories_missing']}\n";
    echo "Not writable: {$verification['directories_not_writable']}\n";
    print_r($verification['details']);
}

// =====================================================
// EXAMPLE 5: Calculate Storage Usage
// =====================================================

echo "\n=== Example 5: Calculate Storage Usage ===\n\n";

$usage = $manager->calculateTenantStorageUsage('soshigh_demo');

if ($usage['success']) {
    echo "Storage Usage Report:\n";
    echo "--------------------\n";
    echo "Total: {$usage['total_formatted']}\n";
    echo "Uploads: " . $manager->formatBytes($usage['usage']['uploads']) . "\n";
    echo "Backups: " . $manager->formatBytes($usage['usage']['backups']) . "\n";
    echo "Logs: " . $manager->formatBytes($usage['usage']['logs']) . "\n";
    echo "Temp: " . $manager->formatBytes($usage['usage']['temp']) . "\n";
    echo "\n";
    echo "Quota: {$usage['quota_formatted']}\n";
    echo "Used: {$usage['percentage']}%\n";
    echo "Available: {$usage['available_formatted']}\n";
    
    if ($usage['percentage'] > 80) {
        echo "\n⚠️ WARNING: Storage quota over 80%!\n";
    }
}

// =====================================================
// EXAMPLE 6: Get and Update Tenant Configuration
// =====================================================

echo "\n=== Example 6: Manage Tenant Configuration ===\n\n";

// Get current config
$config = $manager->getTenantConfig('soshigh_demo');
echo "Current quota: " . $manager->formatBytes($config['storage_quota']) . "\n";

// Update quota
$newQuota = 2147483648; // 2GB
$result = $manager->updateTenantQuota('soshigh_demo', $newQuota);

if ($result['success']) {
    echo "✅ Quota updated to 2GB\n";
}

// Update custom settings
$result = $manager->createTenantConfig('soshigh_demo', [
    'enable_notifications' => true,
    'notification_email' => 'admin@soshigh.edu',
    'auto_backup_enabled' => true,
    'backup_schedule' => 'daily'
]);

if ($result['success']) {
    echo "✅ Configuration updated\n";
}

// =====================================================
// EXAMPLE 7: Integration with School Creation
// =====================================================

echo "\n=== Example 7: School Creation Integration ===\n\n";

/**
 * Function to create a new school with complete setup
 */
function createNewSchool($school_data, $manager, $conn) {
    // 1. Insert school into database
    $stmt = $conn->prepare("
        INSERT INTO tenants 
        (academy_reference, name, contact_email, contact_phone, created_at)
        VALUES (?, ?, ?, ?, NOW())
    ");
    
    $stmt->execute([
        $school_data['academy_reference'],
        $school_data['name'],
        $school_data['email'],
        $school_data['phone']
    ]);
    
    $tenant_id = $school_data['academy_reference'];
    
    // 2. Initialize directory structure
    $init_result = $manager->initializeTenant(
        $tenant_id,
        $school_data['name'],
        [
            'contact_email' => $school_data['email'],
            'contact_phone' => $school_data['phone'],
            'storage_quota' => $school_data['storage_quota'] ?? 1073741824,
            'subscription_tier' => $school_data['subscription_tier'] ?? 'basic'
        ]
    );
    
    if (!$init_result['success']) {
        throw new Exception("Failed to initialize tenant: " . $init_result['error']);
    }
    
    // 3. Create admin user
    $stmt = $conn->prepare("
        INSERT INTO admins 
        (academy_reference, username, password, email, role, created_at)
        VALUES (?, ?, ?, ?, 'admin', NOW())
    ");
    
    $stmt->execute([
        $tenant_id,
        $school_data['admin_username'],
        password_hash($school_data['admin_password'], PASSWORD_DEFAULT),
        $school_data['email']
    ]);
    
    // 4. Verify everything
    $verification = $manager->verifyTenantDirectories($tenant_id);
    
    return [
        'success' => true,
        'tenant_id' => $tenant_id,
        'initialization' => $init_result,
        'verification' => $verification
    ];
}

// Usage example
$new_school = [
    'academy_reference' => 'waterford_2025',
    'name' => 'Waterford Kamhlaba',
    'email' => 'admin@waterford.edu',
    'phone' => '+268-2416-1001',
    'admin_username' => 'waterford_admin',
    'admin_password' => 'secure_password_123',
    'storage_quota' => 5368709120, // 5GB
    'subscription_tier' => 'premium'
];

try {
    $result = createNewSchool($new_school, $manager, $conn);
    echo "✅ School created successfully!\n";
    echo "Tenant ID: {$result['tenant_id']}\n";
    echo "Directories: {$result['initialization']['directories_created']}\n";
} catch (Exception $e) {
    echo "❌ Error: {$e->getMessage()}\n";
}

// =====================================================
// EXAMPLE 8: Bulk Initialize Multiple Tenants
// =====================================================

echo "\n=== Example 8: Bulk Tenant Initialization ===\n\n";

$schools = [
    ['id' => 'school_001', 'name' => 'School One'],
    ['id' => 'school_002', 'name' => 'School Two'],
    ['id' => 'school_003', 'name' => 'School Three']
];

$results = [];

foreach ($schools as $school) {
    $result = $manager->initializeTenant($school['id'], $school['name']);
    $results[] = [
        'school' => $school['name'],
        'success' => $result['success'],
        'message' => $result['message']
    ];
}

echo "Bulk initialization results:\n";
foreach ($results as $r) {
    $status = $r['success'] ? '✅' : '❌';
    echo "{$status} {$r['school']}: {$r['message']}\n";
}

// =====================================================
// EXAMPLE 9: Health Check for All Tenants
// =====================================================

echo "\n=== Example 9: Health Check ===\n\n";

function healthCheckAllTenants($manager, $conn) {
    // Get all tenants from database
    $stmt = $conn->query("SELECT academy_reference, name FROM tenants");
    $tenants = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    $report = [];
    
    foreach ($tenants as $tenant) {
        $tenant_id = $tenant['academy_reference'];
        
        // Verify directories
        $verification = $manager->verifyTenantDirectories($tenant_id);
        
        // Calculate storage
        $storage = $manager->calculateTenantStorageUsage($tenant_id);
        
        $report[] = [
            'tenant_id' => $tenant_id,
            'name' => $tenant['name'],
            'directories_ok' => $verification['directories_ok'] ?? 0,
            'directories_missing' => $verification['directories_missing'] ?? 0,
            'storage_used' => $storage['total_formatted'] ?? 'N/A',
            'storage_percentage' => $storage['percentage'] ?? 0,
            'status' => $verification['success'] ? 'OK' : 'ISSUES'
        ];
    }
    
    return $report;
}

// Run health check
// $health_report = healthCheckAllTenants($manager, $conn);
// 
// foreach ($health_report as $tenant) {
//     echo "{$tenant['status']} - {$tenant['name']}\n";
//     echo "  Storage: {$tenant['storage_used']} ({$tenant['storage_percentage']}%)\n";
//     echo "  Directories: {$tenant['directories_ok']} OK, {$tenant['directories_missing']} missing\n";
// }

// =====================================================
// EXAMPLE 10: Cleanup Script
// =====================================================

echo "\n=== Example 10: Cleanup Old Temp Files ===\n\n";

function cleanupTenantTemp($manager, $tenant_id, $days_old = 7) {
    $tempPath = $manager->getTenantRootPath($tenant_id) . '/temp';
    
    if (!file_exists($tempPath)) {
        return ['deleted' => 0, 'size' => 0];
    }
    
    $cutoffTime = time() - ($days_old * 86400);
    $deleted = 0;
    $totalSize = 0;
    
    $files = scandir($tempPath);
    foreach ($files as $file) {
        if ($file === '.' || $file === '..' || $file === 'index.php') {
            continue;
        }
        
        $fullPath = $tempPath . '/' . $file;
        
        if (is_file($fullPath) && filemtime($fullPath) < $cutoffTime) {
            $size = filesize($fullPath);
            if (unlink($fullPath)) {
                $deleted++;
                $totalSize += $size;
            }
        }
    }
    
    return [
        'deleted' => $deleted,
        'size' => $totalSize,
        'size_formatted' => $manager->formatBytes($totalSize)
    ];
}

// Example usage
// $result = cleanupTenantTemp($manager, 'soshigh_demo', 7);
// echo "Cleaned up {$result['deleted']} files ({$result['size_formatted']})\n";

// =====================================================
// COMPLETE!
// =====================================================

echo "\n=== Examples Complete ===\n";
echo "Check the code above for practical implementations.\n";

