# 📁 Tenant File System - Quick Reference

## ⚡ Quick Setup (2 Minutes)

```sql
-- 1. Import database
SOURCE database/TENANT_FILESYSTEM_TABLES.sql;
```

```php
// 2. Initialize in your PHP
require_once 'includes/tenant_filesystem.php';
$fs = new TenantFileSystem($conn);

// 3. Create tenant directory
$fs->createTenantDirectory($_SESSION['academy_reference']);
```

---

## 📋 Common Operations

### **Upload File**

```php
$result = $fs->uploadFile(
    $_SESSION['academy_reference'],
    $_FILES['document'],
    'documents'
);
```

### **List Files**

```php
$files = $fs->getTenantFiles($tenant_id, 'documents');
foreach ($files['files'] as $file) {
    echo "{$file['name']} - {$file['size_formatted']}\n";
}
```

### **Delete File**

```php
$fs->deleteFile($tenant_id, 'filename.pdf', 'documents');
```

### **Check Storage**

```php
$usage = $fs->getStorageUsage($tenant_id);
echo "{$usage['used_formatted']} / {$usage['quota_formatted']} ({$usage['percentage']}%)";
```

### **Cleanup Temp**

```php
$fs->cleanupTempFiles($tenant_id, 7); // Delete files older than 7 days
```

---

## 🔒 Security Checklist

- ✅ **Path traversal prevention** - Built-in
- ✅ **File type validation** - Per category
- ✅ **File size limits** - Configurable
- ✅ **Storage quotas** - Per tenant
- ✅ **Access control** - Via serve_file.php
- ✅ **Filename sanitization** - Automatic
- ✅ **Logging** - All operations

---

## 📂 File Categories

| Type | Extensions | Max Size |
|------|-----------|----------|
| documents | pdf, doc, docx, xls, xlsx | 10 MB |
| profile_photos | jpg, png, gif, webp | 5 MB |
| payment_receipts | pdf, jpg, png | 5 MB |
| reports | pdf, xls, csv, html | 20 MB |
| backups | sql, zip, tar, gz | 100 MB |
| logs | log, txt | 10 MB |
| temp | * (all) | 20 MB |

---

## 🎯 HTML Forms

### **Single File Upload**

```html
<form method="post" enctype="multipart/form-data">
    <input type="file" name="document" accept=".pdf,.doc,.docx" required>
    <button type="submit">Upload</button>
</form>
```

### **Multiple Files Upload**

```html
<form method="post" enctype="multipart/form-data">
    <input type="file" name="documents[]" multiple>
    <button type="submit">Upload Multiple</button>
</form>
```

### **Profile Photo Upload**

```html
<form method="post" enctype="multipart/form-data">
    <input type="file" name="profile_photo" accept="image/*" required>
    <button type="submit">Upload Photo</button>
</form>
```

---

## 🛠️ Complete Upload Handler

```php
<?php
session_start();
require_once 'config.php';
require_once 'includes/tenant_filesystem.php';

$fs = new TenantFileSystem($conn);

if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_FILES['document'])) {
    $result = $fs->uploadFile(
        $_SESSION['academy_reference'],
        $_FILES['document'],
        'documents',
        'my_custom_name'
    );
    
    echo json_encode($result);
    exit;
}
?>
```

---

## 📊 Get File Info

```php
$files = $fs->getTenantFiles($tenant_id, 'documents', [
    'sort_by' => 'modified',  // name, size, modified
    'sort_order' => 'desc',   // asc, desc
    'limit' => 10,
    'extension' => 'pdf'
]);

foreach ($files['files'] as $file) {
    echo "Name: {$file['name']}\n";
    echo "Size: {$file['size_formatted']}\n";
    echo "Modified: {$file['modified']}\n";
    echo "URL: {$file['url']}\n";
    echo "---\n";
}
```

---

## 🔗 Generate Download Link

```php
$url = $fs->getFileUrl($tenant_id, 'documents', 'report.pdf');
echo "<a href='{$url}' target='_blank'>Download</a>";
```

---

## 📁 Directory Structure

```
tenants/
  soshigh_demo/
    ├── uploads/
    │   ├── documents/
    │   ├── profile_photos/
    │   ├── payment_receipts/
    │   └── reports/
    ├── backups/
    ├── logs/
    ├── temp/
    ├── config.json
    └── .htaccess
```

---

## ⚙️ Configuration

### **Change Default Quota**

```php
// Edit tenants/{tenant_id}/config.json
{
    "storage_quota": 2147483648  // 2GB in bytes
}
```

### **Modify File Size Limits**

```php
// Edit includes/tenant_filesystem.php
private $maxFileSizes = [
    'documents' => 10485760, // 10MB
    // Change as needed
];
```

### **Add Allowed File Type**

```php
// Edit includes/tenant_filesystem.php
private $allowedFileTypes = [
    'documents' => ['pdf', 'doc', 'docx', 'zip'],
    // Add more types
];
```

---

## 🔍 Troubleshooting

### **Permission Issues**

```bash
chmod -R 755 tenants/
chown -R www-data:www-data tenants/
```

### **Storage Quota Exceeded**

```php
// Increase quota
$config['storage_quota'] = 2147483648; // 2GB

// Or cleanup
$fs->cleanupTempFiles($tenant_id, 7);
```

### **File Not Found**

```php
// Validate path
if (!$fs->validateTenantAccess($tenant_id, $filepath)) {
    die('Invalid file path');
}
```

---

## 🧹 Maintenance Commands

```php
// Cleanup temp files (run daily via cron)
$fs->cleanupTempFiles($tenant_id, 7);

// Get storage usage
$usage = $fs->getStorageUsage($tenant_id);
if ($usage['percentage'] > 80) {
    // Send warning email
}
```

---

## ✅ Methods Reference

| Method | Description |
|--------|-------------|
| `createTenantDirectory($tenant_id)` | Create directory structure |
| `getTenantPath($tenant_id, $type)` | Get path for type |
| `uploadFile($tenant_id, $file, $type, $name)` | Upload file |
| `deleteFile($tenant_id, $filepath, $type)` | Delete file |
| `getTenantFiles($tenant_id, $type, $options)` | List files |
| `validateTenantAccess($tenant_id, $filepath)` | Validate access |
| `getStorageUsage($tenant_id)` | Get storage info |
| `cleanupTempFiles($tenant_id, $days)` | Cleanup old files |
| `getFileUrl($tenant_id, $type, $filename)` | Get file URL |

---

## 📚 Full Documentation

See `TENANT_FILESYSTEM_GUIDE.md` for complete documentation.

---

**Ready to use! 🚀**

