<?php
/**
 * TENANT FILE SYSTEM - USAGE EXAMPLES
 * 
 * Examples of how to use the TenantFileSystem class
 */

require_once 'config.php';
require_once 'includes/tenant_filesystem.php';

// Initialize the file system
$fs = new TenantFileSystem($conn);

// =====================================================
// EXAMPLE 1: CREATE TENANT DIRECTORY STRUCTURE
// =====================================================

$tenant_id = 'soshigh_demo';

echo "=== Creating Tenant Directory ===\n";
$result = $fs->createTenantDirectory($tenant_id);

if ($result['success']) {
    echo "✅ Success: " . $result['message'] . "\n";
    echo "Path: " . $result['path'] . "\n";
} else {
    echo "❌ Error: " . $result['error'] . "\n";
}

// =====================================================
// EXAMPLE 2: UPLOAD A FILE
// =====================================================

echo "\n=== Uploading Profile Photo ===\n";

// Simulate file upload from $_FILES
$file = [
    'name' => 'student_photo.jpg',
    'type' => 'image/jpeg',
    'tmp_name' => '/tmp/uploaded_file.jpg', // This would be actual temp path
    'error' => UPLOAD_ERR_OK,
    'size' => 245000 // 245KB
];

$uploadResult = $fs->uploadFile($tenant_id, $file, 'profile_photos');

if ($uploadResult['success']) {
    echo "✅ File uploaded successfully!\n";
    echo "Filename: " . $uploadResult['filename'] . "\n";
    echo "Size: " . $uploadResult['size'] . " bytes\n";
    echo "URL: " . $uploadResult['url'] . "\n";
    echo "Relative path: " . $uploadResult['relative_path'] . "\n";
} else {
    echo "❌ Upload failed: " . $uploadResult['error'] . "\n";
}

// =====================================================
// EXAMPLE 3: UPLOAD DOCUMENT WITH CUSTOM FILENAME
// =====================================================

echo "\n=== Uploading Document with Custom Name ===\n";

$file = [
    'name' => 'report.pdf',
    'type' => 'application/pdf',
    'tmp_name' => '/tmp/report.pdf',
    'error' => UPLOAD_ERR_OK,
    'size' => 1500000 // 1.5MB
];

$uploadResult = $fs->uploadFile($tenant_id, $file, 'documents', 'student_report_2025');

if ($uploadResult['success']) {
    echo "✅ Document uploaded!\n";
    echo "Filename: " . $uploadResult['filename'] . "\n";
}

// =====================================================
// EXAMPLE 4: GET TENANT PATH
// =====================================================

echo "\n=== Getting Tenant Paths ===\n";

// Get base tenant path
$basePath = $fs->getTenantPath($tenant_id);
echo "Base path: {$basePath}\n";

// Get specific type paths
$documentsPath = $fs->getTenantPath($tenant_id, 'documents');
echo "Documents path: {$documentsPath}\n";

$photosPath = $fs->getTenantPath($tenant_id, 'profile_photos');
echo "Photos path: {$photosPath}\n";

// =====================================================
// EXAMPLE 5: LIST ALL FILES FOR A TENANT
// =====================================================

echo "\n=== Listing All Files ===\n";

$filesResult = $fs->getTenantFiles($tenant_id);

if ($filesResult['success']) {
    echo "Total files: " . $filesResult['count'] . "\n";
    
    foreach ($filesResult['files'] as $file) {
        echo "- {$file['name']} ({$file['size_formatted']}) - {$file['type']}\n";
    }
} else {
    echo "❌ Error: " . $filesResult['error'] . "\n";
}

// =====================================================
// EXAMPLE 6: LIST FILES BY TYPE
// =====================================================

echo "\n=== Listing Profile Photos ===\n";

$photosResult = $fs->getTenantFiles($tenant_id, 'profile_photos');

if ($photosResult['success']) {
    echo "Profile photos: " . $photosResult['count'] . "\n";
    
    foreach ($photosResult['files'] as $file) {
        echo "- {$file['name']} (modified: {$file['modified']})\n";
    }
}

// =====================================================
// EXAMPLE 7: LIST FILES WITH OPTIONS
// =====================================================

echo "\n=== Listing Recent Documents (Last 10) ===\n";

$options = [
    'sort_by' => 'modified',
    'sort_order' => 'desc',
    'limit' => 10
];

$recentDocs = $fs->getTenantFiles($tenant_id, 'documents', $options);

if ($recentDocs['success']) {
    foreach ($recentDocs['files'] as $file) {
        echo "- {$file['name']} - {$file['modified']}\n";
    }
}

// =====================================================
// EXAMPLE 8: VALIDATE TENANT ACCESS
// =====================================================

echo "\n=== Validating Tenant Access ===\n";

$filePath = 'uploads/documents/report.pdf';

if ($fs->validateTenantAccess($tenant_id, $filePath)) {
    echo "✅ Access granted to {$filePath}\n";
} else {
    echo "❌ Access denied to {$filePath}\n";
}

// Test path traversal attack
$maliciousPath = '../../../etc/passwd';

if ($fs->validateTenantAccess($tenant_id, $maliciousPath)) {
    echo "⚠️ SECURITY ISSUE: Path traversal not blocked!\n";
} else {
    echo "✅ Path traversal blocked successfully\n";
}

// =====================================================
// EXAMPLE 9: GET STORAGE USAGE
// =====================================================

echo "\n=== Checking Storage Usage ===\n";

$storageInfo = $fs->getStorageUsage($tenant_id);

if ($storageInfo['success']) {
    echo "Used: {$storageInfo['used_formatted']}\n";
    echo "Quota: {$storageInfo['quota_formatted']}\n";
    echo "Available: {$storageInfo['available_formatted']}\n";
    echo "Usage: {$storageInfo['percentage']}%\n";
    
    if ($storageInfo['percentage'] > 80) {
        echo "⚠️ WARNING: Storage quota is over 80%!\n";
    }
}

// =====================================================
// EXAMPLE 10: DELETE A FILE
// =====================================================

echo "\n=== Deleting a File ===\n";

$fileToDelete = 'student_photo_1234567890.jpg';

$deleteResult = $fs->deleteFile($tenant_id, $fileToDelete, 'profile_photos');

if ($deleteResult['success']) {
    echo "✅ File deleted: {$fileToDelete}\n";
} else {
    echo "❌ Delete failed: " . $deleteResult['error'] . "\n";
}

// =====================================================
// EXAMPLE 11: CLEANUP TEMPORARY FILES
// =====================================================

echo "\n=== Cleaning Up Old Temp Files ===\n";

$cleanupResult = $fs->cleanupTempFiles($tenant_id, 7); // Delete files older than 7 days

if ($cleanupResult['success']) {
    echo "✅ Cleaned up {$cleanupResult['deleted']} files\n";
    echo "Space freed: {$cleanupResult['size_freed_formatted']}\n";
}

// =====================================================
// EXAMPLE 12: PRACTICAL FILE UPLOAD HANDLER
// =====================================================

/**
 * Practical example: Handle student document upload
 */
function handleStudentDocumentUpload($student_id, $tenant_id) {
    global $fs;
    
    // Check if file was uploaded
    if (!isset($_FILES['document'])) {
        return ['success' => false, 'error' => 'No file uploaded'];
    }
    
    $file = $_FILES['document'];
    
    // Custom filename with student ID
    $customName = "student_{$student_id}_document";
    
    // Upload the file
    $result = $fs->uploadFile($tenant_id, $file, 'documents', $customName);
    
    if ($result['success']) {
        // Store file info in database
        global $conn;
        $stmt = $conn->prepare("
            INSERT INTO student_documents 
            (student_id, filename, file_path, file_size, uploaded_at)
            VALUES (?, ?, ?, ?, NOW())
        ");
        $stmt->execute([
            $student_id,
            $result['filename'],
            $result['relative_path'],
            $result['size']
        ]);
        
        return [
            'success' => true,
            'file_url' => $result['url'],
            'filename' => $result['filename']
        ];
    }
    
    return $result;
}

// =====================================================
// EXAMPLE 13: PRACTICAL PROFILE PHOTO UPLOAD
// =====================================================

/**
 * Practical example: Handle profile photo with validation
 */
function handleProfilePhotoUpload($user_id, $tenant_id, $user_type = 'student') {
    global $fs, $conn;
    
    if (!isset($_FILES['profile_photo'])) {
        return ['success' => false, 'error' => 'No photo uploaded'];
    }
    
    $file = $_FILES['profile_photo'];
    
    // Validate image
    $imageInfo = @getimagesize($file['tmp_name']);
    if ($imageInfo === false) {
        return ['success' => false, 'error' => 'Invalid image file'];
    }
    
    // Check dimensions
    list($width, $height) = $imageInfo;
    if ($width < 100 || $height < 100) {
        return ['success' => false, 'error' => 'Image too small (minimum 100x100)'];
    }
    
    // Delete old profile photo if exists
    $stmt = $conn->prepare("SELECT profile_photo FROM {$user_type}s WHERE id = ?");
    $stmt->execute([$user_id]);
    $oldPhoto = $stmt->fetchColumn();
    
    if ($oldPhoto) {
        $fs->deleteFile($tenant_id, $oldPhoto, 'profile_photos');
    }
    
    // Upload new photo
    $customName = "{$user_type}_{$user_id}_profile";
    $result = $fs->uploadFile($tenant_id, $file, 'profile_photos', $customName);
    
    if ($result['success']) {
        // Update database
        $stmt = $conn->prepare("UPDATE {$user_type}s SET profile_photo = ? WHERE id = ?");
        $stmt->execute([$result['relative_path'], $user_id]);
        
        return [
            'success' => true,
            'photo_url' => $result['url'],
            'filename' => $result['filename']
        ];
    }
    
    return $result;
}

// =====================================================
// EXAMPLE 14: GENERATE DOWNLOAD LINK
// =====================================================

/**
 * Generate a secure download link for a file
 */
function generateDownloadLink($tenant_id, $type, $filename) {
    global $fs;
    
    // Validate access
    if (!$fs->validateTenantAccess($tenant_id, $type . '/' . $filename)) {
        return false;
    }
    
    return $fs->getFileUrl($tenant_id, $type, $filename);
}

// Usage
echo "\n=== Generating Download Links ===\n";
$downloadUrl = generateDownloadLink($tenant_id, 'documents', 'report.pdf');
echo "Download URL: {$downloadUrl}\n";

// =====================================================
// EXAMPLE 15: BATCH FILE UPLOAD
// =====================================================

/**
 * Upload multiple files at once
 */
function uploadMultipleFiles($tenant_id, $files_array, $type) {
    global $fs;
    
    $results = [
        'success' => [],
        'failed' => []
    ];
    
    foreach ($files_array['tmp_name'] as $index => $tmpName) {
        $file = [
            'name' => $files_array['name'][$index],
            'type' => $files_array['type'][$index],
            'tmp_name' => $tmpName,
            'error' => $files_array['error'][$index],
            'size' => $files_array['size'][$index]
        ];
        
        $result = $fs->uploadFile($tenant_id, $file, $type);
        
        if ($result['success']) {
            $results['success'][] = $result;
        } else {
            $results['failed'][] = [
                'filename' => $file['name'],
                'error' => $result['error']
            ];
        }
    }
    
    return $results;
}

echo "\n=== Batch Upload Example ===\n";
echo "Upload multiple files using: <input type='file' name='documents[]' multiple>\n";

// =====================================================
// EXAMPLE 16: ADMIN PANEL - LIST ALL TENANT FILES
// =====================================================

/**
 * For admin panel: Get all files with statistics
 */
function getAdminFilesList($tenant_id) {
    global $fs;
    
    $stats = [
        'total_files' => 0,
        'total_size' => 0,
        'by_type' => []
    ];
    
    foreach (['documents', 'profile_photos', 'payment_receipts', 'reports'] as $type) {
        $files = $fs->getTenantFiles($tenant_id, $type);
        
        if ($files['success']) {
            $typeSize = array_sum(array_column($files['files'], 'size'));
            
            $stats['by_type'][$type] = [
                'count' => $files['count'],
                'size' => $typeSize,
                'size_formatted' => $fs->formatBytes($typeSize),
                'files' => $files['files']
            ];
            
            $stats['total_files'] += $files['count'];
            $stats['total_size'] += $typeSize;
        }
    }
    
    return $stats;
}

// =====================================================
// COMPLETE!
// =====================================================

echo "\n=== Examples Complete ===\n";
echo "See code above for practical implementations\n";

