# 📤 Upload Guide - Tenant File System

## 🎯 **What to Upload**

Upload these files to implement the complete multi-tenant file system.

---

## 📁 **Files to Upload**

### **1. Core System Files** (Required)

```
📁 includes/
   └── tenant_filesystem.php          ✅ Core class (1,200+ lines)

📁 Root Directory/
   └── serve_file.php                 ✅ Secure file serving script
```

**Upload these to your cPanel File Manager:**
- `includes/tenant_filesystem.php` → `/public_html/includes/`
- `serve_file.php` → `/public_html/`

---

### **2. Database Files** (Required)

```
📁 database/
   ├── TENANT_FILESYSTEM_TABLES_FIXED.sql   ✅ Database schema (USE THIS)
   └── TENANT_FILESYSTEM_TABLES.sql         ⚠️ Original (has foreign key issues)
```

**⚠️ IMPORTANT: Use the FIXED version!**

**Import in phpMyAdmin:**
1. Go to phpMyAdmin
2. Select your database
3. Go to **Import** tab
4. Choose: `database/TENANT_FILESYSTEM_TABLES_FIXED.sql`
5. Click **Go**

Creates 5 tables:
- `file_operations_log`
- `file_access_log`
- `tenant_files`
- `file_shares`
- `tenant_storage_quotas`

---

### **3. Documentation Files** (Recommended)

```
📁 Root Directory/
   ├── TENANT_FILESYSTEM_GUIDE.md                    ✅ Complete guide
   ├── TENANT_FILESYSTEM_QUICK_REFERENCE.md          ✅ Quick reference
   ├── TENANT_FILESYSTEM_USAGE_EXAMPLES.php          ✅ Code examples
   ├── TENANT_FILESYSTEM_IMPLEMENTATION_SUMMARY.md   ✅ Summary
   └── UPLOAD_TENANT_FILESYSTEM.md                   ✅ This file
```

These are for your reference - upload to root or a `docs/` folder.

---

### **4. Test File** (Optional, for testing only)

```
📁 Root Directory/
   └── test_tenant_filesystem.php     ✅ Automated test suite
```

**⚠️ Upload temporarily for testing, then DELETE after verification!**

---

## 🚀 **Quick Upload Steps**

### **Step 1: Upload Core Files (2 minutes)**

**Via cPanel File Manager:**

1. Login to cPanel
2. Open **File Manager**
3. Navigate to `/public_html/includes/`
4. Click **Upload**
5. Upload: `tenant_filesystem.php`
6. Navigate to `/public_html/`
7. Upload: `serve_file.php`

---

### **Step 2: Import Database (2 minutes)**

**Via phpMyAdmin:**

1. Login to phpMyAdmin
2. Select database: `melane_multi_tenant_skoloi`
3. Click **Import** tab
4. Click **Choose File**
5. Select: `TENANT_FILESYSTEM_TABLES_FIXED.sql` ⚠️ Use FIXED version!
6. Click **Go**
7. Wait for success message

**Expected Result:**
```
✅ Import has been successfully finished
✅ 5 queries executed
✅ 5 tables created
```

---

### **Step 3: Set Permissions (1 minute)**

**Create tenants directory:**

1. In cPanel File Manager
2. Navigate to `/public_html/`
3. Click **+ Folder**
4. Name it: `tenants`
5. Right-click → **Change Permissions**
6. Set to: `755` (rwxr-xr-x)

**Or via SSH:**
```bash
cd /home/melane/public_html/test.melanegroup.com/Multi-Tanent/
mkdir -p tenants
chmod 755 tenants
```

---

### **Step 4: Test Installation (2 minutes)**

**Upload test file:**
1. Upload `test_tenant_filesystem.php` to root
2. Visit: `https://test.melanegroup.com/Multi-Tanent/test_tenant_filesystem.php`
3. Run all tests
4. Verify all pass (green checkmarks)
5. **DELETE test file after verification**

---

## 📋 **Verification Checklist**

After upload, verify:

### **Files Present:**
```bash
- [ ] includes/tenant_filesystem.php exists
- [ ] serve_file.php exists
- [ ] tenants/ directory exists with 755 permissions
```

### **Database:**
```sql
-- Run in phpMyAdmin SQL tab:
SHOW TABLES LIKE 'file_%';
SHOW TABLES LIKE 'tenant_storage%';

-- Should return:
-- file_operations_log
-- file_access_log
-- tenant_files
-- file_shares
-- tenant_storage_quotas
```

### **Test Suite:**
```
- [ ] All 10 tests pass
- [ ] No red error messages
- [ ] Green success indicators
- [ ] Test files cleaned up
```

---

## 🔧 **Configuration After Upload**

### **Optional: Adjust File Limits**

Edit `includes/tenant_filesystem.php`:

**Line ~40-50: Allowed File Types**
```php
private $allowedFileTypes = [
    'documents' => ['pdf', 'doc', 'docx', 'xls', 'xlsx', 'txt', 'csv'],
    // Add more as needed
];
```

**Line ~55-65: Max File Sizes**
```php
private $maxFileSizes = [
    'documents' => 10485760, // 10MB
    // Adjust as needed
];
```

---

## 🎯 **Integration with Your App**

### **In School Registration** (`super_admin/schools/create.php`)

Add after school is created:

```php
// Create tenant file system
require_once '../includes/tenant_filesystem.php';
$fs = new TenantFileSystem($conn);
$fs->createTenantDirectory($academy_reference);
```

### **In Student Document Upload**

```php
require_once 'includes/tenant_filesystem.php';
$fs = new TenantFileSystem($conn);

if (isset($_FILES['document'])) {
    $result = $fs->uploadFile(
        $_SESSION['academy_reference'],
        $_FILES['document'],
        'documents'
    );
    
    if ($result['success']) {
        // Save to database
        $stmt = $conn->prepare("
            INSERT INTO student_documents 
            (student_id, filename, file_path, uploaded_at)
            VALUES (?, ?, ?, NOW())
        ");
        $stmt->execute([
            $student_id,
            $result['filename'],
            $result['relative_path']
        ]);
    }
}
```

### **In Profile Photo Upload**

```php
require_once 'includes/tenant_filesystem.php';
$fs = new TenantFileSystem($conn);

if (isset($_FILES['profile_photo'])) {
    // Delete old photo
    if ($oldPhotoPath) {
        $fs->deleteFile($_SESSION['academy_reference'], $oldPhotoPath);
    }
    
    // Upload new
    $result = $fs->uploadFile(
        $_SESSION['academy_reference'],
        $_FILES['profile_photo'],
        'profile_photos',
        "student_{$student_id}_profile"
    );
    
    if ($result['success']) {
        // Update database
        $conn->prepare("UPDATE students SET profile_photo = ? WHERE id = ?")
             ->execute([$result['relative_path'], $student_id]);
    }
}
```

---

## 🧹 **Post-Upload Cleanup**

### **Files to DELETE After Testing:**

```
❌ test_tenant_filesystem.php
❌ TENANT_FILESYSTEM_USAGE_EXAMPLES.php (optional, keep for reference)
```

### **Files to KEEP:**

```
✅ includes/tenant_filesystem.php
✅ serve_file.php
✅ TENANT_FILESYSTEM_GUIDE.md (reference)
✅ TENANT_FILESYSTEM_QUICK_REFERENCE.md (reference)
```

---

## 🔄 **Maintenance Setup**

### **Create Cron Job for Cleanup**

In cPanel → Cron Jobs:

**Daily Cleanup (Recommended):**
```
0 2 * * * /usr/bin/php /home/melane/public_html/test.melanegroup.com/Multi-Tanent/cron/cleanup_temp_files.php
```

**Create file:** `cron/cleanup_temp_files.php`
```php
<?php
require_once '../config.php';
require_once '../includes/tenant_filesystem.php';

$fs = new TenantFileSystem($conn);

// Get all tenants
$stmt = $conn->query("SELECT DISTINCT academy_reference FROM tenants");
$tenants = $stmt->fetchAll(PDO::FETCH_COLUMN);

foreach ($tenants as $tenant_id) {
    $result = $fs->cleanupTempFiles($tenant_id, 7);
    if ($result['success']) {
        echo "Cleaned {$result['deleted']} files for {$tenant_id}\n";
    }
}
```

---

## 📊 **Database Verification Queries**

After import, run these in phpMyAdmin:

```sql
-- Check tables created
SHOW TABLES LIKE 'file%';
SHOW TABLES LIKE 'tenant_storage%';

-- Expected: 5 tables

-- Check table structure
DESCRIBE file_operations_log;
DESCRIBE file_access_log;
DESCRIBE tenant_storage_quotas;

-- Create storage quota for existing tenants
INSERT IGNORE INTO tenant_storage_quotas (tenant_id, storage_quota)
SELECT DISTINCT academy_reference, 1073741824
FROM tenants
WHERE academy_reference IS NOT NULL;

-- Verify quotas created
SELECT * FROM tenant_storage_quotas;
```

---

## 🐛 **Troubleshooting**

### **Issue: "Permission denied"**

```bash
chmod -R 755 tenants/
chown -R [your_user]:nogroup tenants/
```

### **Issue: "Class TenantFileSystem not found"**

- Verify `tenant_filesystem.php` uploaded to `includes/`
- Check file has no PHP syntax errors
- Verify `require_once` path is correct

### **Issue: "Table doesn't exist"**

- Re-import `TENANT_FILESYSTEM_TABLES.sql`
- Check database name is correct
- Verify you're using correct database in phpMyAdmin

### **Issue: "Failed to create directory"**

- Check `tenants/` folder exists
- Verify permissions (755)
- Check disk space available

---

## ✅ **Success Indicators**

You'll know it's working when:

1. ✅ Test suite shows all green checkmarks
2. ✅ `tenants/test_XXXXXXXX/` folder created during test
3. ✅ Test file uploaded successfully
4. ✅ Storage usage calculated correctly
5. ✅ Database logs show operations
6. ✅ Security tests block path traversal
7. ✅ Invalid file types rejected
8. ✅ Test files cleaned up automatically

---

## 📚 **Documentation Access**

After upload, access documentation at:

```
https://test.melanegroup.com/Multi-Tanent/TENANT_FILESYSTEM_GUIDE.md
https://test.melanegroup.com/Multi-Tanent/TENANT_FILESYSTEM_QUICK_REFERENCE.md
```

---

## 🎯 **Final Checklist**

Before going live:

- [ ] Core files uploaded (`tenant_filesystem.php`, `serve_file.php`)
- [ ] Database imported (5 tables created)
- [ ] `tenants/` directory created with 755 permissions
- [ ] Test suite run and all tests pass
- [ ] Test file deleted from server
- [ ] Integration code added to your app
- [ ] Cron job set up (optional but recommended)
- [ ] Documentation reviewed
- [ ] Storage quotas created for existing tenants

---

## 🚀 **You're Ready!**

Total upload time: **10 minutes**  
Total files: **2 core + 1 database + docs**  
Production ready: **✅ YES**

The multi-tenant file system is now installed and ready for use!

---

**Need help?** Check:
- `TENANT_FILESYSTEM_GUIDE.md` - Complete documentation
- `TENANT_FILESYSTEM_QUICK_REFERENCE.md` - Quick reference
- `test_tenant_filesystem.php` - Run tests to diagnose issues

---

*File: UPLOAD_TENANT_FILESYSTEM.md*  
*Last Updated: 2025*  
*Status: Complete Upload Guide*

