<?php
/**
 * Verify Manual Payment for Student Application
 * Allows admins to verify uploaded payment receipts
 */

session_start();
require_once '../../includes/functions.php';
require_once '../../includes/school_context.php';
requireAdminLogin();

$pdo = getDB();
$academy_reference = getCurrentSchool();

if (!$academy_reference) {
    die("Error: No school selected.");
}

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    header('Location: list.php');
    exit;
}

$application_id = (int)($_POST['application_id'] ?? 0);
$action = $_POST['action'] ?? '';

// Verify application belongs to this school
$stmt = $pdo->prepare("
    SELECT * FROM student_applications 
    WHERE id = ? AND academy_reference = ?
");
$stmt->execute([$application_id, $academy_reference]);
$app = $stmt->fetch();

if (!$app) {
    die("Application not found or access denied.");
}

$admin_id = $_SESSION['admin_id'] ?? null;

if ($action === 'approve_payment') {
    // Approve manual payment
    
    try {
        $pdo->beginTransaction();
        
        // Update payment status
        $stmt = $pdo->prepare("
            UPDATE student_applications 
            SET payment_status = 'paid',
                payment_method = 'manual',
                payment_date = NOW(),
                admin_notes = CONCAT(COALESCE(admin_notes, ''), '\nPayment manually verified by admin on ', NOW())
            WHERE id = ?
        ");
        $stmt->execute([$application_id]);
        
        $pdo->commit();
        
        $_SESSION['success'] = 'Payment verified successfully! You can now approve the application.';
        header('Location: view.php?id=' . $application_id);
    } catch (Exception $e) {
        $pdo->rollBack();
        $_SESSION['error'] = 'Error verifying payment: ' . $e->getMessage();
        header('Location: view.php?id=' . $application_id);
    }
    
} elseif ($action === 'reject_payment') {
    // Reject manual payment
    
    $rejection_reason = trim($_POST['payment_rejection_reason'] ?? '');
    
    if (empty($rejection_reason)) {
        $_SESSION['error'] = 'Rejection reason is required.';
        header('Location: view.php?id=' . $application_id);
        exit;
    }
    
    try {
        $pdo->beginTransaction();
        
        // Update payment status
        $stmt = $pdo->prepare("
            UPDATE student_applications 
            SET payment_status = 'failed',
                admin_notes = CONCAT(COALESCE(admin_notes, ''), '\nPayment rejected on ', NOW(), ': ', ?)
            WHERE id = ?
        ");
        $stmt->execute([$rejection_reason, $application_id]);
        
        // Send notification email
        $subject = "Payment Verification Failed";
        $message = "
            <!DOCTYPE html>
            <html>
            <head>
                <style>
                    body { font-family: Arial, sans-serif; line-height: 1.6; color: #333; }
                    .container { max-width: 600px; margin: 0 auto; padding: 20px; }
                    .header { background: #dc3545; color: white; padding: 20px; text-align: center; border-radius: 10px 10px 0 0; }
                    .content { background: #ffffff; padding: 30px; border: 1px solid #e0e0e0; }
                    .reason-box { background: #f8d7da; border-left: 4px solid #dc3545; padding: 15px; margin: 20px 0; border-radius: 5px; }
                </style>
            </head>
            <body>
                <div class='container'>
                    <div class='header'>
                        <h2>Payment Verification Failed</h2>
                    </div>
                    <div class='content'>
                        <p>Dear {$app['full_name']},</p>
                        <p>We were unable to verify your payment. Please review the reason below and submit a new payment.</p>
                        
                        <div class='reason-box'>
                            <strong>Reason:</strong><br>
                            {$rejection_reason}
                        </div>
                        
                        <p>Please contact the school administration if you have questions.</p>
                        
                        <p>Best regards,<br><strong>School Administration</strong></p>
                    </div>
                </div>
            </body>
            </html>
        ";
        
        $headers = "MIME-Version: 1.0\r\n";
        $headers .= "Content-type:text/html;charset=UTF-8\r\n";
        $headers .= "From: School Administration <noreply@melanegroup.com>\r\n";
        
        mail($app['email'], $subject, $message, $headers);
        
        $pdo->commit();
        
        $_SESSION['success'] = 'Payment rejected and applicant notified.';
        header('Location: view.php?id=' . $application_id);
    } catch (Exception $e) {
        $pdo->rollBack();
        $_SESSION['error'] = 'Error rejecting payment: ' . $e->getMessage();
        header('Location: view.php?id=' . $application_id);
    }
    
} else {
    $_SESSION['error'] = 'Invalid action.';
    header('Location: view.php?id=' . $application_id);
}

exit;

