<?php
/**
 * Manage School Grades
 * Allows admin to select which grades their school offers
 */

require_once '../includes/functions.php';
require_once '../includes/school_context.php';
requireAdminLogin();

$pdo = getDB();
$academy_reference = getCurrentSchool();

if (!$academy_reference) {
    die("Error: No school selected. Please select a school first.");
}

$success = '';
$error = '';

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['save_grades'])) {
    $selected_grades = $_POST['grades'] ?? [];
    
    try {
        $pdo->beginTransaction();
        
        // Delete all existing grade selections for this school
        $stmt = $pdo->prepare("DELETE FROM school_grades WHERE academy_reference = ?");
        $stmt->execute([$academy_reference]);
        
        // Insert selected grades
        if (!empty($selected_grades)) {
            $stmt = $pdo->prepare("
                INSERT INTO school_grades (academy_reference, grade_id, is_active, created_by)
                VALUES (?, ?, 1, ?)
            ");
            
            foreach ($selected_grades as $grade_id) {
                $stmt->execute([$academy_reference, intval($grade_id), $_SESSION['user_id'] ?? null]);
            }
        }
        
        $pdo->commit();
        $success = "Grade configuration saved successfully! Selected " . count($selected_grades) . " grade(s).";
        
    } catch (Exception $e) {
        $pdo->rollBack();
        $error = "Error saving grades: " . $e->getMessage();
    }
}

// Get all available grades
$all_grades = getGrades();

// Separate primary and secondary grades
$primary_grades = array_filter($all_grades, function($g) {
    return preg_match('/^Grade [4-7]$/i', $g['name']);
});

$secondary_grades = array_filter($all_grades, function($g) {
    return preg_match('/^Form [1-5]$/i', $g['name']);
});

// Get currently selected grades for this school
$stmt = $pdo->prepare("
    SELECT grade_id FROM school_grades 
    WHERE academy_reference = ? AND is_active = 1
");
$stmt->execute([$academy_reference]);
$selected_grade_ids = array_column($stmt->fetchAll(), 'grade_id');

// Get school name
$school_name = getSchoolName() ?? $academy_reference;

?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Manage Grades - <?= htmlspecialchars($school_name) ?></title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.0/font/bootstrap-icons.min.css" rel="stylesheet">
    <style>
        .grade-selector {
            cursor: pointer;
            transition: all 0.3s;
            border: 2px solid #dee2e6;
        }
        .grade-selector:hover {
            border-color: #0d6efd;
            transform: translateY(-2px);
            box-shadow: 0 4px 10px rgba(0,0,0,0.1);
        }
        .grade-selector.selected {
            border-color: #198754;
            background: linear-gradient(135deg, rgba(25,135,84,0.1) 0%, rgba(25,135,84,0.05) 100%);
        }
        .grade-selector input[type="checkbox"] {
            width: 24px;
            height: 24px;
        }
        .section-header {
            border-left: 4px solid;
            padding-left: 15px;
        }
        .primary-section { border-left-color: #198754; }
        .secondary-section { border-left-color: #0d6efd; }
    </style>
</head>
<body class="bg-light">
    
    <!-- Header -->
    <?php include 'partials/header.php'; ?>
    
    <div class="container py-4">
        <div class="row mb-4">
            <div class="col-12">
                <div class="d-flex justify-content-between align-items-center">
                    <div>
                        <h2><i class="bi bi-sliders me-2"></i>Manage School Grades</h2>
                        <p class="text-muted">Select which grades your school offers</p>
                    </div>
                    <a href="configure_subjects.php" class="btn btn-outline-primary">
                        <i class="bi bi-gear me-2"></i>Configure Subjects
                    </a>
                </div>
            </div>
        </div>
        
        <!-- Alerts -->
        <?php if ($success): ?>
        <div class="alert alert-success alert-dismissible fade show">
            <i class="bi bi-check-circle-fill me-2"></i><?= htmlspecialchars($success) ?>
            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
        </div>
        <?php endif; ?>
        
        <?php if ($error): ?>
        <div class="alert alert-danger alert-dismissible fade show">
            <i class="bi bi-exclamation-triangle-fill me-2"></i><?= htmlspecialchars($error) ?>
            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
        </div>
        <?php endif; ?>
        
        <!-- School Info -->
        <div class="card shadow-sm mb-4">
            <div class="card-body">
                <h5 class="card-title">
                    <i class="bi bi-building me-2"></i>
                    <?= htmlspecialchars($school_name) ?>
                </h5>
                <p class="text-muted mb-0">
                    <small>Academy Reference: <code><?= htmlspecialchars($academy_reference) ?></code></small>
                </p>
            </div>
        </div>
        
        <!-- Grade Selection Form -->
        <form method="POST" id="gradeForm">
            
            <!-- Primary Grades Section -->
            <?php if (!empty($primary_grades)): ?>
            <div class="card shadow-sm mb-4">
                <div class="card-header bg-success text-white">
                    <h5 class="mb-0">
                        <i class="bi bi-book me-2"></i>Primary School Grades (Grade 4-7)
                    </h5>
                </div>
                <div class="card-body">
                    <p class="text-muted mb-3">
                        <i class="bi bi-info-circle me-1"></i>
                        Select the primary grades your school offers (typically ages 9-13)
                    </p>
                    <div class="row g-3">
                        <?php foreach ($primary_grades as $grade): ?>
                        <?php $is_selected = in_array($grade['id'], $selected_grade_ids); ?>
                        <div class="col-md-6 col-lg-3">
                            <div class="grade-selector card h-100 <?= $is_selected ? 'selected' : '' ?>"
                                 onclick="toggleGrade(this, <?= $grade['id'] ?>)">
                                <div class="card-body text-center">
                                    <div class="form-check d-flex align-items-center justify-content-center">
                                        <input class="form-check-input me-3" 
                                               type="checkbox" 
                                               name="grades[]" 
                                               value="<?= $grade['id'] ?>"
                                               id="grade_<?= $grade['id'] ?>"
                                               <?= $is_selected ? 'checked' : '' ?>
                                               onclick="event.stopPropagation()">
                                        <label class="form-check-label" for="grade_<?= $grade['id'] ?>">
                                            <h5 class="mb-0"><?= htmlspecialchars($grade['name']) ?></h5>
                                        </label>
                                    </div>
                                    <?php if ($is_selected): ?>
                                    <div class="mt-2">
                                        <span class="badge bg-success">Active</span>
                                    </div>
                                    <?php endif; ?>
                                </div>
                            </div>
                        </div>
                        <?php endforeach; ?>
                    </div>
                </div>
            </div>
            <?php endif; ?>
            
            <!-- Secondary Grades Section -->
            <?php if (!empty($secondary_grades)): ?>
            <div class="card shadow-sm mb-4">
                <div class="card-header bg-primary text-white">
                    <h5 class="mb-0">
                        <i class="bi bi-mortarboard-fill me-2"></i>Secondary School Grades (Form 1-5)
                    </h5>
                </div>
                <div class="card-body">
                    <p class="text-muted mb-3">
                        <i class="bi bi-info-circle me-1"></i>
                        Select the secondary grades your school offers (typically ages 13-18)
                    </p>
                    <div class="row g-3">
                        <?php foreach ($secondary_grades as $grade): ?>
                        <?php $is_selected = in_array($grade['id'], $selected_grade_ids); ?>
                        <div class="col-md-6 col-lg-3">
                            <div class="grade-selector card h-100 <?= $is_selected ? 'selected' : '' ?>"
                                 onclick="toggleGrade(this, <?= $grade['id'] ?>)">
                                <div class="card-body text-center">
                                    <div class="form-check d-flex align-items-center justify-content-center">
                                        <input class="form-check-input me-3" 
                                               type="checkbox" 
                                               name="grades[]" 
                                               value="<?= $grade['id'] ?>"
                                               id="grade_<?= $grade['id'] ?>"
                                               <?= $is_selected ? 'checked' : '' ?>
                                               onclick="event.stopPropagation()">
                                        <label class="form-check-label" for="grade_<?= $grade['id'] ?>">
                                            <h5 class="mb-0"><?= htmlspecialchars($grade['name']) ?></h5>
                                        </label>
                                    </div>
                                    <?php if ($is_selected): ?>
                                    <div class="mt-2">
                                        <span class="badge bg-success">Active</span>
                                    </div>
                                    <?php endif; ?>
                                </div>
                            </div>
                        </div>
                        <?php endforeach; ?>
                    </div>
                </div>
            </div>
            <?php endif; ?>
            
            <!-- Summary & Actions -->
            <div class="card shadow-sm">
                <div class="card-body">
                    <div class="row align-items-center">
                        <div class="col-md-8">
                            <h6 class="mb-1">Selected Grades Summary</h6>
                            <p class="mb-0 text-muted">
                                <span id="selected_count"><?= count($selected_grade_ids) ?></span> grade(s) currently active
                            </p>
                        </div>
                        <div class="col-md-4 text-end">
                            <button type="button" class="btn btn-outline-secondary me-2" onclick="selectAll()">
                                <i class="bi bi-check-all me-1"></i>Select All
                            </button>
                            <button type="button" class="btn btn-outline-secondary me-2" onclick="deselectAll()">
                                <i class="bi bi-x-lg me-1"></i>Clear All
                            </button>
                            <button type="submit" name="save_grades" class="btn btn-success">
                                <i class="bi bi-save me-1"></i>Save Configuration
                            </button>
                        </div>
                    </div>
                </div>
            </div>
            
        </form>
        
        <!-- Help Section -->
        <div class="card shadow-sm mt-4">
            <div class="card-header bg-info text-white">
                <h6 class="mb-0"><i class="bi bi-question-circle me-2"></i>How This Works</h6>
            </div>
            <div class="card-body">
                <ul class="mb-0">
                    <li><strong>Select grades your school offers</strong> - Only selected grades will appear in your admin interface</li>
                    <li><strong>Configure subjects per grade</strong> - After saving, go to "Configure Subjects" to set which subjects are available</li>
                    <li><strong>Parent experience</strong> - Parents will only see grades you've enabled</li>
                    <li><strong>Student registration</strong> - Only enabled grades available during registration</li>
                    <li><strong>Flexibility</strong> - Each school can offer different grade levels</li>
                </ul>
            </div>
        </div>
        
        <!-- Examples -->
        <div class="row mt-4">
            <div class="col-md-4">
                <div class="card shadow-sm">
                    <div class="card-header bg-light">
                        <strong>Example: Primary School</strong>
                    </div>
                    <div class="card-body">
                        <p class="small mb-2">Select:</p>
                        <ul class="small mb-0">
                            <li>✓ Grade 4</li>
                            <li>✓ Grade 5</li>
                            <li>✓ Grade 6</li>
                            <li>✓ Grade 7</li>
                        </ul>
                    </div>
                </div>
            </div>
            <div class="col-md-4">
                <div class="card shadow-sm">
                    <div class="card-header bg-light">
                        <strong>Example: Secondary School</strong>
                    </div>
                    <div class="card-body">
                        <p class="small mb-2">Select:</p>
                        <ul class="small mb-0">
                            <li>✓ Form 1</li>
                            <li>✓ Form 2</li>
                            <li>✓ Form 3</li>
                            <li>✓ Form 4</li>
                            <li>✓ Form 5</li>
                        </ul>
                    </div>
                </div>
            </div>
            <div class="col-md-4">
                <div class="card shadow-sm">
                    <div class="card-header bg-light">
                        <strong>Example: Combined School</strong>
                    </div>
                    <div class="card-body">
                        <p class="small mb-2">Select:</p>
                        <ul class="small mb-0">
                            <li>✓ Grade 6</li>
                            <li>✓ Grade 7</li>
                            <li>✓ Form 1</li>
                            <li>✓ Form 2</li>
                            <li>✓ Form 3</li>
                        </ul>
                    </div>
                </div>
            </div>
        </div>
        
    </div>
    
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        // Toggle grade selection
        function toggleGrade(card, gradeId) {
            const checkbox = card.querySelector('input[type="checkbox"]');
            checkbox.checked = !checkbox.checked;
            
            if (checkbox.checked) {
                card.classList.add('selected');
            } else {
                card.classList.remove('selected');
            }
            
            updateCount();
        }
        
        // Select all grades
        function selectAll() {
            document.querySelectorAll('.grade-selector input[type="checkbox"]').forEach(cb => {
                cb.checked = true;
                cb.closest('.grade-selector').classList.add('selected');
            });
            updateCount();
        }
        
        // Deselect all grades
        function deselectAll() {
            document.querySelectorAll('.grade-selector input[type="checkbox"]').forEach(cb => {
                cb.checked = false;
                cb.closest('.grade-selector').classList.remove('selected');
            });
            updateCount();
        }
        
        // Update selected count
        function updateCount() {
            const count = document.querySelectorAll('.grade-selector input[type="checkbox"]:checked').length;
            document.getElementById('selected_count').textContent = count;
        }
        
        // Initialize
        document.addEventListener('DOMContentLoaded', function() {
            updateCount();
        });
    </script>
</body>
</html>

