<?php
require_once '../includes/functions.php';
require_once '../includes/monitoring_system.php';

// Check admin authentication
if (!isAdminLoggedIn()) {
    header('Location: login.php');
    exit;
}

// Initialize monitoring
$monitoring = new MonitoringSystem();

// Get monitoring data
$system_stats = $monitoring->getSystemStatistics(24);
$recent_errors = [];
$active_alerts = [];
$health_checks = [];

try {
    $pdo = getDB();
    
    // Get recent errors
    $stmt = $pdo->query("
        SELECT * FROM error_logs 
        ORDER BY created_at DESC 
        LIMIT 20
    ");
    $recent_errors = $stmt->fetchAll();
    
    // Get active alerts
    $stmt = $pdo->query("
        SELECT * FROM alerts 
        WHERE status = 'ACTIVE' 
        ORDER BY created_at DESC
    ");
    $active_alerts = $stmt->fetchAll();
    
    // Get latest health checks
    $stmt = $pdo->query("
        SELECT check_name, status, check_value, message, created_at
        FROM health_checks 
        WHERE created_at > DATE_SUB(NOW(), INTERVAL 1 HOUR)
        ORDER BY created_at DESC
    ");
    $health_checks = $stmt->fetchAll();
    
} catch (Exception $e) {
    $error = "Failed to load monitoring data: " . $e->getMessage();
}

// Handle actions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (!validateCSRFToken($_POST['csrf_token'] ?? '')) {
        $error = 'Invalid security token';
    } else {
        $action = $_POST['action'] ?? '';
        
        switch ($action) {
            case 'run_health_checks':
                $monitoring->performHealthChecks();
                $success = "Health checks completed";
                break;
                
            case 'acknowledge_alert':
                $alert_id = (int)($_POST['alert_id'] ?? 0);
                if ($alert_id > 0) {
                    $stmt = $pdo->prepare("
                        UPDATE alerts 
                        SET status = 'ACKNOWLEDGED', acknowledged_by = ?, acknowledged_at = NOW() 
                        WHERE id = ?
                    ");
                    $stmt->execute([$_SESSION['admin_id'] ?? 1, $alert_id]);
                    $success = "Alert acknowledged";
                }
                break;
        }
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>System Monitoring - <?php echo APP_NAME; ?></title>    
    <!-- Favicon -->
    <link rel="icon" type="image/svg+xml" href="../images/img-melanesys-logo.svg">
    <link rel="alternate icon" href="../images/img-melanesys-logo.svg">
    <link rel="apple-touch-icon" href="../images/img-melanesys-logo.svg">
    
    
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.3/font/bootstrap-icons.min.css" rel="stylesheet">
    <script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
</head>
<body class="bg-light">
    <div class="container-fluid">
        <div class="row">
            <!-- Sidebar -->
            <div class="col-md-3 col-lg-2 d-md-block bg-dark sidebar collapse">
                <div class="position-sticky pt-3">
                    <ul class="nav flex-column">
                        <li class="nav-item">
                            <a class="nav-link text-white" href="dashboard.php">
                                <i class="bi bi-speedometer2 me-2"></i>Dashboard
                            </a>
                        </li>
                        <li class="nav-item">
                            <a class="nav-link text-white" href="backup_management.php">
                                <i class="bi bi-archive me-2"></i>Backups
                            </a>
                        </li>
                        <li class="nav-item">
                            <a class="nav-link text-white active" href="monitoring_dashboard.php">
                                <i class="bi bi-graph-up me-2"></i>Monitoring
                            </a>
                        </li>
                        <li class="nav-item">
                            <a class="nav-link text-white" href="logout.php">
                                <i class="bi bi-box-arrow-right me-2"></i>Logout
                            </a>
                        </li>
                    </ul>
                </div>
            </div>

            <!-- Main content -->
            <main class="col-md-9 ms-sm-auto col-lg-10 px-md-4">
                <div class="d-flex justify-content-between flex-wrap flex-md-nowrap align-items-center pt-3 pb-2 mb-3 border-bottom">
                    <h1 class="h2">
                        <i class="bi bi-graph-up me-2"></i>System Monitoring
                    </h1>
                    <div class="btn-toolbar mb-2 mb-md-0">
                        <form method="POST" class="d-inline">
                            <input type="hidden" name="csrf_token" value="<?php echo generateCSRFToken(); ?>">
                            <input type="hidden" name="action" value="run_health_checks">
                            <button type="submit" class="btn btn-outline-primary">
                                <i class="bi bi-heart-pulse me-2"></i>Run Health Checks
                            </button>
                        </form>
                    </div>
                </div>

                <?php if (isset($success)): ?>
                    <div class="alert alert-success alert-dismissible fade show">
                        <i class="bi bi-check-circle me-2"></i><?php echo htmlspecialchars($success); ?>
                        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                    </div>
                <?php endif; ?>

                <?php if (isset($error)): ?>
                    <div class="alert alert-danger alert-dismissible fade show">
                        <i class="bi bi-exclamation-triangle me-2"></i><?php echo htmlspecialchars($error); ?>
                        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                    </div>
                <?php endif; ?>

                <!-- System Overview -->
                <div class="row mb-4">
                    <div class="col-md-3">
                        <div class="card text-white bg-primary">
                            <div class="card-body">
                                <div class="d-flex justify-content-between">
                                    <div>
                                        <h4 class="card-title">System Health</h4>
                                        <p class="card-text">
                                            <?php 
                                            $healthy_checks = array_filter($health_checks, function($check) { return $check['status'] === 'HEALTHY'; });
                                            $total_checks = count($health_checks);
                                            echo $total_checks > 0 ? round((count($healthy_checks) / $total_checks) * 100) : 0;
                                            ?>%
                                        </p>
                                    </div>
                                    <div class="align-self-center">
                                        <i class="bi bi-heart-pulse fs-1"></i>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                    
                    <div class="col-md-3">
                        <div class="card text-white bg-warning">
                            <div class="card-body">
                                <div class="d-flex justify-content-between">
                                    <div>
                                        <h4 class="card-title">Active Alerts</h4>
                                        <p class="card-text"><?php echo count($active_alerts); ?></p>
                                    </div>
                                    <div class="align-self-center">
                                        <i class="bi bi-exclamation-triangle fs-1"></i>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                    
                    <div class="col-md-3">
                        <div class="card text-white bg-danger">
                            <div class="card-body">
                                <div class="d-flex justify-content-between">
                                    <div>
                                        <h4 class="card-title">Recent Errors</h4>
                                        <p class="card-text"><?php echo count($recent_errors); ?></p>
                                    </div>
                                    <div class="align-self-center">
                                        <i class="bi bi-bug fs-1"></i>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                    
                    <div class="col-md-3">
                        <div class="card text-white bg-success">
                            <div class="card-body">
                                <div class="d-flex justify-content-between">
                                    <div>
                                        <h4 class="card-title">Uptime</h4>
                                        <p class="card-text">99.9%</p>
                                    </div>
                                    <div class="align-self-center">
                                        <i class="bi bi-clock-history fs-1"></i>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Health Checks -->
                <div class="row mb-4">
                    <div class="col-12">
                        <div class="card">
                            <div class="card-header">
                                <h5 class="card-title mb-0">
                                    <i class="bi bi-heart-pulse me-2"></i>Health Checks
                                </h5>
                            </div>
                            <div class="card-body">
                                <?php if (empty($health_checks)): ?>
                                    <div class="text-center py-4">
                                        <i class="bi bi-heart-pulse fs-1 text-muted"></i>
                                        <p class="text-muted mt-2">No recent health checks</p>
                                    </div>
                                <?php else: ?>
                                    <div class="table-responsive">
                                        <table class="table table-sm">
                                            <thead>
                                                <tr>
                                                    <th>Check</th>
                                                    <th>Status</th>
                                                    <th>Value</th>
                                                    <th>Message</th>
                                                    <th>Time</th>
                                                </tr>
                                            </thead>
                                            <tbody>
                                                <?php foreach ($health_checks as $check): ?>
                                                    <tr>
                                                        <td><?php echo htmlspecialchars($check['check_name']); ?></td>
                                                        <td>
                                                            <?php
                                                            $status_class = '';
                                                            $status_icon = '';
                                                            switch ($check['status']) {
                                                                case 'HEALTHY':
                                                                    $status_class = 'success';
                                                                    $status_icon = 'check-circle';
                                                                    break;
                                                                case 'WARNING':
                                                                    $status_class = 'warning';
                                                                    $status_icon = 'exclamation-triangle';
                                                                    break;
                                                                case 'CRITICAL':
                                                                    $status_class = 'danger';
                                                                    $status_icon = 'x-circle';
                                                                    break;
                                                                default:
                                                                    $status_class = 'secondary';
                                                                    $status_icon = 'question-circle';
                                                            }
                                                            ?>
                                                            <span class="badge bg-<?php echo $status_class; ?>">
                                                                <i class="bi bi-<?php echo $status_icon; ?> me-1"></i>
                                                                <?php echo $check['status']; ?>
                                                            </span>
                                                        </td>
                                                        <td><?php echo $check['check_value']; ?></td>
                                                        <td><?php echo htmlspecialchars($check['message']); ?></td>
                                                        <td><?php echo $check['created_at']; ?></td>
                                                    </tr>
                                                <?php endforeach; ?>
                                            </tbody>
                                        </table>
                                    </div>
                                <?php endif; ?>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Active Alerts -->
                <div class="row mb-4">
                    <div class="col-12">
                        <div class="card">
                            <div class="card-header">
                                <h5 class="card-title mb-0">
                                    <i class="bi bi-exclamation-triangle me-2"></i>Active Alerts
                                </h5>
                            </div>
                            <div class="card-body">
                                <?php if (empty($active_alerts)): ?>
                                    <div class="text-center py-4">
                                        <i class="bi bi-check-circle fs-1 text-success"></i>
                                        <p class="text-muted mt-2">No active alerts</p>
                                    </div>
                                <?php else: ?>
                                    <div class="table-responsive">
                                        <table class="table table-sm">
                                            <thead>
                                                <tr>
                                                    <th>Type</th>
                                                    <th>Severity</th>
                                                    <th>Title</th>
                                                    <th>Message</th>
                                                    <th>Created</th>
                                                    <th>Actions</th>
                                                </tr>
                                            </thead>
                                            <tbody>
                                                <?php foreach ($active_alerts as $alert): ?>
                                                    <tr>
                                                        <td><?php echo htmlspecialchars($alert['alert_type']); ?></td>
                                                        <td>
                                                            <?php
                                                            $severity_class = '';
                                                            switch ($alert['severity']) {
                                                                case 'LOW':
                                                                    $severity_class = 'info';
                                                                    break;
                                                                case 'MEDIUM':
                                                                    $severity_class = 'warning';
                                                                    break;
                                                                case 'HIGH':
                                                                    $severity_class = 'danger';
                                                                    break;
                                                                case 'CRITICAL':
                                                                    $severity_class = 'dark';
                                                                    break;
                                                            }
                                                            ?>
                                                            <span class="badge bg-<?php echo $severity_class; ?>">
                                                                <?php echo $alert['severity']; ?>
                                                            </span>
                                                        </td>
                                                        <td><?php echo htmlspecialchars($alert['title']); ?></td>
                                                        <td><?php echo htmlspecialchars($alert['message']); ?></td>
                                                        <td><?php echo $alert['created_at']; ?></td>
                                                        <td>
                                                            <form method="POST" class="d-inline">
                                                                <input type="hidden" name="csrf_token" value="<?php echo generateCSRFToken(); ?>">
                                                                <input type="hidden" name="action" value="acknowledge_alert">
                                                                <input type="hidden" name="alert_id" value="<?php echo $alert['id']; ?>">
                                                                <button type="submit" class="btn btn-sm btn-outline-primary">
                                                                    <i class="bi bi-check"></i> Acknowledge
                                                                </button>
                                                            </form>
                                                        </td>
                                                    </tr>
                                                <?php endforeach; ?>
                                            </tbody>
                                        </table>
                                    </div>
                                <?php endif; ?>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Recent Errors -->
                <div class="row">
                    <div class="col-12">
                        <div class="card">
                            <div class="card-header">
                                <h5 class="card-title mb-0">
                                    <i class="bi bi-bug me-2"></i>Recent Errors
                                </h5>
                            </div>
                            <div class="card-body">
                                <?php if (empty($recent_errors)): ?>
                                    <div class="text-center py-4">
                                        <i class="bi bi-check-circle fs-1 text-success"></i>
                                        <p class="text-muted mt-2">No recent errors</p>
                                    </div>
                                <?php else: ?>
                                    <div class="table-responsive">
                                        <table class="table table-sm">
                                            <thead>
                                                <tr>
                                                    <th>Type</th>
                                                    <th>Message</th>
                                                    <th>File</th>
                                                    <th>Line</th>
                                                    <th>Time</th>
                                                </tr>
                                            </thead>
                                            <tbody>
                                                <?php foreach ($recent_errors as $error): ?>
                                                    <tr>
                                                        <td>
                                                            <span class="badge bg-danger">
                                                                <?php echo htmlspecialchars($error['error_type']); ?>
                                                            </span>
                                                        </td>
                                                        <td>
                                                            <small><?php echo htmlspecialchars(substr($error['error_message'], 0, 100)); ?>...</small>
                                                        </td>
                                                        <td>
                                                            <small><?php echo htmlspecialchars(basename($error['error_file'])); ?></small>
                                                        </td>
                                                        <td><?php echo $error['error_line']; ?></td>
                                                        <td><?php echo $error['created_at']; ?></td>
                                                    </tr>
                                                <?php endforeach; ?>
                                            </tbody>
                                        </table>
                                    </div>
                                <?php endif; ?>
                            </div>
                        </div>
                    </div>
                </div>
            </main>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        // Auto-refresh every 30 seconds
        setTimeout(function() {
            location.reload();
        }, 30000);
    </script>
</body>
</html>
