<?php
/**
 * Moodle API Configuration
 * Configure Moodle URL, API token, and integration settings
 */

session_start();
require_once '../../includes/functions.php';
require_once '../../includes/school_context.php';
require_once '../../includes/favicon_helper.php';
requireAdminLogin();

if (!isSuperAdmin()) {
    header('Location: index.php');
    exit;
}

$pdo = getDB();
$academy_reference = getCurrentSchool();
$schoolName = getSchoolName();

// Get current settings
$stmt = $pdo->prepare("SELECT setting_key, setting_value FROM tenant_settings WHERE academy_reference = ?");
$stmt->execute([$academy_reference]);
$settings = [];
while ($row = $stmt->fetch()) {
    $settings[$row['setting_key']] = $row['setting_value'];
}

$success = '';
$error = '';

// Handle AJAX test connection
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'test_moodle') {
    header('Content-Type: application/json');
    
    $moodleUrl = trim($_POST['moodle_url'] ?? '');
    $moodleToken = trim($_POST['moodle_token'] ?? '');
    
    if (empty($moodleUrl) || empty($moodleToken)) {
        echo json_encode(['success' => false, 'message' => 'Moodle URL and token are required']);
        exit;
    }
    
    try {
        $testUrl = rtrim($moodleUrl, '/') . '/webservice/rest/server.php';
        $params = [
            'wstoken' => $moodleToken,
            'wsfunction' => 'core_webservice_get_site_info',
            'moodlewsrestformat' => 'json'
        ];
        
        $ch = curl_init($testUrl . '?' . http_build_query($params));
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_TIMEOUT, 10);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
        
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        
        if ($httpCode === 200) {
            $data = json_decode($response, true);
            if (isset($data['sitename'])) {
                echo json_encode([
                    'success' => true,
                    'message' => 'Successfully connected to Moodle!',
                    'site_name' => $data['sitename'],
                    'version' => $data['release'] ?? 'Unknown'
                ]);
            } elseif (isset($data['exception'])) {
                echo json_encode(['success' => false, 'message' => 'Moodle Error: ' . ($data['message'] ?? $data['exception'])]);
            } else {
                echo json_encode(['success' => false, 'message' => 'Invalid response from Moodle']);
            }
        } else {
            echo json_encode(['success' => false, 'message' => 'Failed to connect to Moodle (HTTP ' . $httpCode . ')']);
        }
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => 'Connection error: ' . $e->getMessage()]);
    }
    exit;
}

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST' && !isset($_POST['action'])) {
    $moodleUrl = trim($_POST['moodle_url'] ?? '');
    $moodleToken = trim($_POST['moodle_token'] ?? '');
    $moodleServiceName = trim($_POST['moodle_service_name'] ?? 'moodle_mobile_app');
    $moodleEnabled = isset($_POST['moodle_enabled']) ? 1 : 0;
    
    try {
        $pdo->beginTransaction();
        
        $settingsToSave = [
            'moodle_url' => $moodleUrl,
            'moodle_token' => $moodleToken,
            'moodle_service_name' => $moodleServiceName,
            'moodle_enabled' => $moodleEnabled
        ];
        
        foreach ($settingsToSave as $key => $value) {
            $stmt = $pdo->prepare("
                INSERT INTO tenant_settings (academy_reference, setting_key, setting_value)
                VALUES (?, ?, ?)
                ON DUPLICATE KEY UPDATE setting_value = VALUES(setting_value)
            ");
            $stmt->execute([$academy_reference, $key, $value]);
        }
        
        $pdo->commit();
        $success = 'Moodle settings saved successfully!';
        
        // Refresh settings
        $settings = $settingsToSave;
        
        // Redirect to test connection page after 2 seconds
        header("refresh:2;url=test_connection.php");
        
    } catch (Exception $e) {
        $pdo->rollBack();
        $error = 'Error saving settings: ' . $e->getMessage();
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Moodle API Settings - <?= htmlspecialchars($schoolName) ?></title>
    <?php echoFavicon($academy_reference); ?>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.0/font/bootstrap-icons.min.css" rel="stylesheet">
</head>
<body class="bg-light">
    <div class="container mt-4 mb-5">
        <!-- Header -->
        <div class="d-flex justify-content-between align-items-center mb-4">
            <div>
                <h2><i class="bi bi-gear text-primary"></i> Moodle API Configuration</h2>
                <p class="text-muted mb-0">Configure Moodle integration for <?= htmlspecialchars($schoolName) ?></p>
            </div>
            <div>
                <a href="index.php" class="btn btn-outline-secondary">
                    <i class="bi bi-arrow-left"></i> Moodle Hub
                </a>
            </div>
        </div>

        <!-- Alerts -->
        <?php if ($success): ?>
        <div class="alert alert-success alert-dismissible fade show">
            <i class="bi bi-check-circle me-2"></i><?= htmlspecialchars($success) ?>
            <br><small>Redirecting to test connection in 2 seconds...</small>
            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
        </div>
        <?php endif; ?>
        
        <?php if ($error): ?>
        <div class="alert alert-danger alert-dismissible fade show">
            <i class="bi bi-exclamation-triangle me-2"></i><?= htmlspecialchars($error) ?>
            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
        </div>
        <?php endif; ?>

        <!-- Test Result -->
        <div id="testResult"></div>

        <!-- Configuration Form -->
        <div class="card shadow-sm">
            <div class="card-header bg-primary text-white">
                <h5 class="mb-0"><i class="bi bi-mortarboard me-2"></i>Moodle API Settings</h5>
            </div>
            <div class="card-body">
                <form method="POST" id="moodleForm">
                    <div class="mb-4">
                        <div class="form-check form-switch">
                            <input class="form-check-input" type="checkbox" id="moodle_enabled" name="moodle_enabled" 
                                   <?= ($settings['moodle_enabled'] ?? 0) ? 'checked' : '' ?>>
                            <label class="form-check-label" for="moodle_enabled">
                                <strong>Enable Moodle Integration</strong>
                                <small class="d-block text-muted">Sync students with your Moodle LMS</small>
                            </label>
                        </div>
                    </div>

                    <div id="moodle-settings" style="display: <?= ($settings['moodle_enabled'] ?? 0) ? 'block' : 'none' ?>;">
                        <div class="mb-3">
                            <label for="moodle_url" class="form-label">
                                <i class="bi bi-link-45deg me-1"></i>Moodle URL *
                            </label>
                            <input type="url" class="form-control" id="moodle_url" name="moodle_url" 
                                   value="<?= htmlspecialchars($settings['moodle_url'] ?? '') ?>"
                                   placeholder="https://your-moodle-site.com" required>
                            <small class="text-muted">Your Moodle site URL (e.g., https://lms.yourschool.com)</small>
                        </div>

                        <div class="mb-3">
                            <label for="moodle_token" class="form-label">
                                <i class="bi bi-key me-1"></i>Moodle API Token *
                            </label>
                            <input type="password" class="form-control" id="moodle_token" name="moodle_token" 
                                   value="<?= htmlspecialchars($settings['moodle_token'] ?? '') ?>"
                                   placeholder="Enter your Moodle web service token" required>
                            <small class="text-muted">Web service token from Moodle Site Administration</small>
                        </div>

                        <div class="mb-3">
                            <label for="moodle_service_name" class="form-label">
                                Service Name
                            </label>
                            <input type="text" class="form-control" id="moodle_service_name" name="moodle_service_name" 
                                   value="<?= htmlspecialchars($settings['moodle_service_name'] ?? 'moodle_mobile_app') ?>">
                            <small class="text-muted">Usually "moodle_mobile_app" (default)</small>
                        </div>

                        <div class="d-flex gap-2 mb-3">
                            <button type="button" id="testMoodleBtn" class="btn btn-info">
                                <i class="bi bi-plug"></i> Test Connection
                            </button>
                            <button type="submit" class="btn btn-primary">
                                <i class="bi bi-save"></i> Save Settings
                            </button>
                        </div>
                    </div>
                </form>

                <!-- Instructions -->
                <div class="alert alert-info mt-4">
                    <h6><i class="bi bi-info-circle me-2"></i>How to get your Moodle API token:</h6>
                    <ol class="mb-0">
                        <li>Login to your Moodle site as administrator</li>
                        <li>Go to <strong>Site administration → Server → Web services → Manage tokens</strong></li>
                        <li>Create a token for a user with appropriate permissions</li>
                        <li>Select service: <code>moodle_mobile_app</code> or create custom service</li>
                        <li>Copy the generated token and paste it above</li>
                    </ol>
                </div>

                <!-- Quick Actions -->
                <div class="mt-4">
                    <h6>Quick Actions:</h6>
                    <div class="btn-group" role="group">
                        <a href="test_connection.php" class="btn btn-outline-success">
                            <i class="bi bi-check-circle"></i> Test Connection
                        </a>
                        <a href="configure_mapping.php" class="btn btn-outline-primary">
                            <i class="bi bi-diagram-3"></i> Configure Mappings
                        </a>
                        <a href="/Multi-Tanent/test_moodle_functions.php?key=test123" class="btn btn-outline-info" target="_blank">
                            <i class="bi bi-clipboard-check"></i> Test API Functions
                        </a>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        // Toggle Moodle settings visibility
        document.getElementById('moodle_enabled').addEventListener('change', function() {
            document.getElementById('moodle-settings').style.display = this.checked ? 'block' : 'none';
        });

        // Test Moodle connection
        document.getElementById('testMoodleBtn').addEventListener('click', function() {
            const btn = this;
            const resultDiv = document.getElementById('testResult');
            const moodleUrl = document.getElementById('moodle_url').value;
            const moodleToken = document.getElementById('moodle_token').value;

            if (!moodleUrl || !moodleToken) {
                resultDiv.innerHTML = '<div class="alert alert-warning">Please enter Moodle URL and token first.</div>';
                return;
            }

            btn.disabled = true;
            btn.innerHTML = '<span class="spinner-border spinner-border-sm me-2"></span>Testing...';
            resultDiv.innerHTML = '';

            fetch('', {
                method: 'POST',
                headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
                body: new URLSearchParams({
                    action: 'test_moodle',
                    moodle_url: moodleUrl,
                    moodle_token: moodleToken
                })
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    resultDiv.innerHTML = `
                        <div class="alert alert-success alert-dismissible fade show">
                            <i class="bi bi-check-circle me-2"></i>
                            <strong>${data.message}</strong><br>
                            <small>Site: ${data.site_name} | Version: ${data.version}</small>
                            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                        </div>
                    `;
                } else {
                    resultDiv.innerHTML = `
                        <div class="alert alert-danger alert-dismissible fade show">
                            <i class="bi bi-x-circle me-2"></i>${data.message}
                            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                        </div>
                    `;
                }
            })
            .catch(error => {
                resultDiv.innerHTML = `
                    <div class="alert alert-danger alert-dismissible fade show">
                        <i class="bi bi-x-circle me-2"></i>Error: ${error.message}
                        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                    </div>
                `;
            })
            .finally(() => {
                btn.disabled = false;
                btn.innerHTML = '<i class="bi bi-plug"></i> Test Connection';
            });
        });
    </script>
</body>
</html>

