<?php
/**
 * Generic Upload Handler (Tenant-Aware)
 * 
 * Handles all file uploads in admin panel
 * 
 * POST Parameters:
 *   - file: The uploaded file
 *   - type: document|photo|receipt|report
 *   - related_id: Student ID, Payment ID, etc (optional)
 *   - related_type: student|payment|report (optional)
 *   - description: File description (optional)
 *   - tags: Comma-separated tags (optional)
 */

session_start();
require_once '../config.php';
require_once '../includes/functions.php';
require_once '../includes/tenant_directory_manager.php';
require_once '../includes/tenant_file_upload.php';

// Check authentication
if (!isAdminLoggedIn()) {
    http_response_code(401);
    echo json_encode(['success' => false, 'error' => 'Unauthorized']);
    exit;
}

// Check tenant
if (!isset($_SESSION['academy_reference'])) {
    http_response_code(400);
    echo json_encode(['success' => false, 'error' => 'No tenant context']);
    exit;
}

$tenant_id = $_SESSION['academy_reference'];

// Validate request
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['success' => false, 'error' => 'Method not allowed']);
    exit;
}

if (!isset($_FILES['file'])) {
    http_response_code(400);
    echo json_encode(['success' => false, 'error' => 'No file uploaded']);
    exit;
}

$upload_type = $_POST['type'] ?? 'document';
$related_id = isset($_POST['related_id']) ? (int)$_POST['related_id'] : null;
$related_type = $_POST['related_type'] ?? null;
$description = $_POST['description'] ?? null;
$tags = $_POST['tags'] ?? null;

try {
    // Database connection
    $conn = getDB();
    
    // Initialize upload handler
    $dirManager = new TenantDirectoryManager($conn);
    $uploader = new TenantFileUpload($conn, $dirManager);
    
    // Prepare metadata
    $metadata = [
        'description' => $description,
        'tags' => $tags,
        'related_id' => $related_id,
        'related_type' => $related_type
    ];
    
    // Upload based on type
    switch ($upload_type) {
        case 'photo':
            if (!$related_id) {
                throw new Exception("Student ID required for photo upload");
            }
            $result = $uploader->uploadProfilePhoto($tenant_id, $related_id, $_FILES['file']);
            break;
            
        case 'receipt':
            if (!$related_id) {
                throw new Exception("Payment ID required for receipt upload");
            }
            $result = $uploader->uploadPaymentReceipt($tenant_id, $related_id, $_FILES['file']);
            break;
            
        case 'report':
            $report_type = $_POST['report_type'] ?? 'general';
            $result = $uploader->uploadReport($tenant_id, $report_type, $_FILES['file'], $metadata);
            break;
            
        case 'document':
        default:
            $result = $uploader->uploadDocument($tenant_id, $_FILES['file'], $metadata);
            break;
    }
    
    // Return response
    header('Content-Type: application/json');
    echo json_encode($result);
    
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode(['success' => false, 'error' => $e->getMessage()]);
}

