<?php
/**
 * Email Configuration Checker
 * This script helps diagnose email delivery issues
 */

require_once 'config.php';

// Only allow access from localhost or specific IP
$allowed_ips = ['127.0.0.1', '::1', 'localhost'];
if (!in_array($_SERVER['REMOTE_ADDR'], $allowed_ips) && !isset($_GET['allow'])) {
    die('Access denied. This tool is for local testing only.');
}

?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Email Configuration Checker</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.0/font/bootstrap-icons.min.css" rel="stylesheet">
</head>
<body class="bg-light">
    <div class="container py-5">
        <div class="row justify-content-center">
            <div class="col-lg-8">
                <div class="card shadow">
                    <div class="card-header bg-primary text-white">
                        <h4 class="mb-0"><i class="bi bi-envelope-check me-2"></i>Email Configuration Checker</h4>
                    </div>
                    <div class="card-body">
                        
                        <h5 class="mb-3">SMTP Configuration Status</h5>
                        
                        <?php
                        $checks = [];
                        
                        // Check if constants are defined
                        $checks['SMTP_HOST'] = defined('SMTP_HOST') && !empty(SMTP_HOST);
                        $checks['SMTP_PORT'] = defined('SMTP_PORT') && !empty(SMTP_PORT);
                        $checks['SMTP_USERNAME'] = defined('SMTP_USERNAME') && !empty(SMTP_USERNAME);
                        $checks['SMTP_PASSWORD'] = defined('SMTP_PASSWORD') && !empty(SMTP_PASSWORD);
                        $checks['SMTP_ENCRYPTION'] = defined('SMTP_ENCRYPTION') && !empty(SMTP_ENCRYPTION);
                        $checks['SMTP_FROM_EMAIL'] = defined('SMTP_FROM_EMAIL') && !empty(SMTP_FROM_EMAIL);
                        
                        // Check if password is not placeholder
                        $password_configured = defined('SMTP_PASSWORD') && SMTP_PASSWORD !== 'your-email-password';
                        
                        $all_configured = array_reduce($checks, fn($carry, $item) => $carry && $item, true) && $password_configured;
                        ?>
                        
                        <div class="table-responsive">
                            <table class="table table-bordered">
                                <thead>
                                    <tr>
                                        <th>Setting</th>
                                        <th>Value</th>
                                        <th>Status</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <tr>
                                        <td><strong>SMTP_HOST</strong></td>
                                        <td><code><?= defined('SMTP_HOST') ? htmlspecialchars(SMTP_HOST) : 'Not defined' ?></code></td>
                                        <td>
                                            <?php if ($checks['SMTP_HOST']): ?>
                                                <span class="badge bg-success">✓ OK</span>
                                            <?php else: ?>
                                                <span class="badge bg-danger">✗ Missing</span>
                                            <?php endif; ?>
                                        </td>
                                    </tr>
                                    <tr>
                                        <td><strong>SMTP_PORT</strong></td>
                                        <td><code><?= defined('SMTP_PORT') ? htmlspecialchars(SMTP_PORT) : 'Not defined' ?></code></td>
                                        <td>
                                            <?php if ($checks['SMTP_PORT']): ?>
                                                <span class="badge bg-success">✓ OK</span>
                                            <?php else: ?>
                                                <span class="badge bg-danger">✗ Missing</span>
                                            <?php endif; ?>
                                        </td>
                                    </tr>
                                    <tr>
                                        <td><strong>SMTP_USERNAME</strong></td>
                                        <td><code><?= defined('SMTP_USERNAME') ? htmlspecialchars(SMTP_USERNAME) : 'Not defined' ?></code></td>
                                        <td>
                                            <?php if ($checks['SMTP_USERNAME']): ?>
                                                <span class="badge bg-success">✓ OK</span>
                                            <?php else: ?>
                                                <span class="badge bg-danger">✗ Missing</span>
                                            <?php endif; ?>
                                        </td>
                                    </tr>
                                    <tr>
                                        <td><strong>SMTP_PASSWORD</strong></td>
                                        <td>
                                            <?php if (!defined('SMTP_PASSWORD') || empty(SMTP_PASSWORD)): ?>
                                                <span class="text-danger">Not set</span>
                                            <?php elseif (SMTP_PASSWORD === 'your-email-password'): ?>
                                                <span class="text-warning">⚠️ Using placeholder password</span>
                                            <?php else: ?>
                                                <code>••••••••••</code> <small class="text-success">(Configured)</small>
                                            <?php endif; ?>
                                        </td>
                                        <td>
                                            <?php if ($password_configured): ?>
                                                <span class="badge bg-success">✓ OK</span>
                                            <?php else: ?>
                                                <span class="badge bg-danger">✗ Not Configured</span>
                                            <?php endif; ?>
                                        </td>
                                    </tr>
                                    <tr>
                                        <td><strong>SMTP_ENCRYPTION</strong></td>
                                        <td><code><?= defined('SMTP_ENCRYPTION') ? htmlspecialchars(SMTP_ENCRYPTION) : 'Not defined' ?></code></td>
                                        <td>
                                            <?php if ($checks['SMTP_ENCRYPTION']): ?>
                                                <span class="badge bg-success">✓ OK</span>
                                            <?php else: ?>
                                                <span class="badge bg-danger">✗ Missing</span>
                                            <?php endif; ?>
                                        </td>
                                    </tr>
                                    <tr>
                                        <td><strong>SMTP_FROM_EMAIL</strong></td>
                                        <td><code><?= defined('SMTP_FROM_EMAIL') ? htmlspecialchars(SMTP_FROM_EMAIL) : 'Not defined' ?></code></td>
                                        <td>
                                            <?php if ($checks['SMTP_FROM_EMAIL']): ?>
                                                <span class="badge bg-success">✓ OK</span>
                                            <?php else: ?>
                                                <span class="badge bg-danger">✗ Missing</span>
                                            <?php endif; ?>
                                        </td>
                                    </tr>
                                </tbody>
                            </table>
                        </div>
                        
                        <div class="mt-4">
                            <?php if ($all_configured): ?>
                                <div class="alert alert-success">
                                    <h5><i class="bi bi-check-circle me-2"></i>SMTP is Properly Configured!</h5>
                                    <p class="mb-0">Your SMTP settings look good. Emails should be sent via SMTP.</p>
                                </div>
                            <?php else: ?>
                                <div class="alert alert-warning">
                                    <h5><i class="bi bi-exclamation-triangle me-2"></i>SMTP Not Fully Configured</h5>
                                    <p>Some SMTP settings are missing or incorrect. The system will use <strong>PHP mail()</strong> as a fallback.</p>
                                    <hr>
                                    <p class="mb-0"><strong>To fix this:</strong></p>
                                    <ol class="mb-0">
                                        <li>Open <code>config.php</code></li>
                                        <li>Update <code>SMTP_PASSWORD</code> with your actual email password</li>
                                        <li>Verify other SMTP settings match your email provider</li>
                                        <li>Refresh this page to check again</li>
                                    </ol>
                                </div>
                            <?php endif; ?>
                        </div>
                        
                        <div class="mt-4">
                            <h5>Email Method Being Used:</h5>
                            <?php if ($all_configured): ?>
                                <div class="alert alert-info">
                                    <i class="bi bi-server me-2"></i><strong>SMTP</strong> - Using configured SMTP server for better deliverability
                                </div>
                            <?php else: ?>
                                <div class="alert alert-secondary">
                                    <i class="bi bi-envelope me-2"></i><strong>PHP mail()</strong> - Using server's mail function (may have deliverability issues)
                                </div>
                            <?php endif; ?>
                        </div>
                        
                        <div class="mt-4">
                            <h5>What This Means:</h5>
                            <div class="alert alert-light">
                                <p class="mb-2"><strong>Good News:</strong> Email sending will still work, even without SMTP!</p>
                                <ul class="mb-0">
                                    <li>With SMTP: Higher deliverability, fewer spam issues</li>
                                    <li>Without SMTP: Uses PHP mail(), may go to spam folder</li>
                                    <li>Registration always works: Direct verification button appears if email fails</li>
                                </ul>
                            </div>
                        </div>
                        
                        <div class="mt-4">
                            <h5>Quick Fix Steps:</h5>
                            <ol>
                                <li>Open your <code>config.php</code> file</li>
                                <li>Find line: <code>define('SMTP_PASSWORD', 'your-email-password');</code></li>
                                <li>Replace with: <code>define('SMTP_PASSWORD', 'YourActualPassword');</code></li>
                                <li>Save and upload the file</li>
                                <li>Test registration again</li>
                            </ol>
                        </div>
                        
                        <div class="mt-4 d-grid gap-2">
                            <a href="register_student.php?school=KINE" class="btn btn-primary">
                                <i class="bi bi-arrow-left me-2"></i>Test Registration
                            </a>
                        </div>
                        
                    </div>
                </div>
                
                <div class="text-center mt-3">
                    <small class="text-muted">
                        This diagnostic tool helps identify email configuration issues.<br>
                        Delete this file after fixing email issues for security.
                    </small>
                </div>
            </div>
        </div>
    </div>
    
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>

