<?php
/**
 * Database Optimization Cron Job
 * Cleans up database and optimizes performance
 */

require_once __DIR__ . '/cron_utils.php';

// Log file
$logFile = __DIR__ . '/../logs/db_optimization.log';

logMessage("=== DATABASE OPTIMIZATION STARTED ===");

try {
    $pdo = getCronDB();
    if (!$pdo) {
        throw new Exception("Database connection failed");
    }
    
    // Get database size before optimization
    $stmt = $pdo->query("SELECT 
        ROUND(SUM(data_length + index_length) / 1024 / 1024, 2) AS 'DB Size in MB'
        FROM information_schema.tables 
        WHERE table_schema = DATABASE()");
    $result = $stmt->fetch();
    $dbSizeBefore = $result['DB Size in MB'];
    logMessage("Database size before optimization: {$dbSizeBefore}MB");
    
    // Optimize all tables
    $stmt = $pdo->query("SHOW TABLES");
    $tables = $stmt->fetchAll(PDO::FETCH_COLUMN);
    $optimizedTables = 0;
    
    foreach ($tables as $table) {
        try {
            $pdo->exec("OPTIMIZE TABLE `$table`");
            logMessage("Optimized table: $table");
            $optimizedTables++;
        } catch (Exception $e) {
            logMessage("WARNING: Failed to optimize table $table: " . $e->getMessage());
        }
    }
    
    // Clean up old email verification tokens (older than 7 days)
    $stmt = $pdo->prepare("DELETE FROM students 
        WHERE email_verification_expires < NOW() - INTERVAL 7 DAY 
        AND email_verified = 0");
    $stmt->execute();
    $deletedTokens = $stmt->rowCount();
    logMessage("Cleaned up expired email verification tokens: $deletedTokens");
    
    // Clean up old temp passwords (older than 1 day)
    $stmt = $pdo->prepare("UPDATE students 
        SET temp_password = NULL 
        WHERE temp_password IS NOT NULL 
        AND created_at < NOW() - INTERVAL 1 DAY");
    $stmt->execute();
    $cleanedPasswords = $stmt->rowCount();
    logMessage("Cleaned up old temp passwords: $cleanedPasswords");
    
    // Analyze tables for better query performance
    $analyzedTables = 0;
    foreach ($tables as $table) {
        try {
            $pdo->exec("ANALYZE TABLE `$table`");
            $analyzedTables++;
        } catch (Exception $e) {
            logMessage("WARNING: Failed to analyze table $table: " . $e->getMessage());
        }
    }
    
    // Check for duplicate students (same email)
    $stmt = $pdo->query("SELECT email, COUNT(*) as count 
        FROM students 
        WHERE email IS NOT NULL 
        GROUP BY email 
        HAVING count > 1");
    $duplicates = $stmt->fetchAll();
    
    if (count($duplicates) > 0) {
        logMessage("WARNING: Found " . count($duplicates) . " duplicate email addresses");
        foreach ($duplicates as $duplicate) {
            logMessage("Duplicate email: {$duplicate['email']} (count: {$duplicate['count']})");
        }
    } else {
        logMessage("No duplicate email addresses found");
    }
    
    // Check for orphaned records
    $stmt = $pdo->query("SELECT COUNT(*) as count FROM students WHERE grade_id IS NULL");
    $orphaned = $stmt->fetch();
    if ($orphaned['count'] > 0) {
        logMessage("WARNING: Found {$orphaned['count']} students without grade_id");
    }
    
    // Get database size after optimization
    $stmt = $pdo->query("SELECT 
        ROUND(SUM(data_length + index_length) / 1024 / 1024, 2) AS 'DB Size in MB'
        FROM information_schema.tables 
        WHERE table_schema = DATABASE()");
    $result = $stmt->fetch();
    $dbSizeAfter = $result['DB Size in MB'];
    
    $sizeReduction = $dbSizeBefore - $dbSizeAfter;
    $reductionPercent = $dbSizeBefore > 0 ? round(($sizeReduction / $dbSizeBefore) * 100, 2) : 0;
    
    logMessage("Database optimization completed:");
    logMessage("- Tables optimized: $optimizedTables");
    logMessage("- Tables analyzed: $analyzedTables");
    logMessage("- Expired tokens cleaned: $deletedTokens");
    logMessage("- Temp passwords cleaned: $cleanedPasswords");
    logMessage("- Database size before: {$dbSizeBefore}MB");
    logMessage("- Database size after: {$dbSizeAfter}MB");
    logMessage("- Size reduction: {$sizeReduction}MB ({$reductionPercent}%)");
    
    // Check table status
    $stmt = $pdo->query("SHOW TABLE STATUS");
    $tableStatus = $stmt->fetchAll();
    
    $totalRows = 0;
    $totalDataLength = 0;
    $totalIndexLength = 0;
    
    foreach ($tableStatus as $table) {
        $totalRows += $table['Rows'];
        $totalDataLength += $table['Data_length'];
        $totalIndexLength += $table['Index_length'];
    }
    
    logMessage("Database statistics:");
    logMessage("- Total rows: " . number_format($totalRows));
    logMessage("- Total data length: " . round($totalDataLength / 1024 / 1024, 2) . "MB");
    logMessage("- Total index length: " . round($totalIndexLength / 1024 / 1024, 2) . "MB");
    
} catch (Exception $e) {
    logMessage("ERROR: Database optimization failed: " . $e->getMessage());
    exit(1);
}

logMessage("=== DATABASE OPTIMIZATION COMPLETED ===");
?>
