<?php
/**
 * Backup Verification Cron Job
 * Ensures backups are working and notifies if issues
 */

require_once __DIR__ . '/cron_utils.php';

// Set execution time limit
set_time_limit(300); // 5 minutes

// Log file
$logFile = __DIR__ . '/../logs/backup_verification.log';

// Ensure logs directory exists
if (!is_dir(__DIR__ . '/../logs')) {
    mkdir(__DIR__ . '/../logs', 0755, true);
}



function sendAlert($subject, $message) {
    $adminEmail = 'admin@' . ($_SERVER['SERVER_NAME'] ?? 'localhost');
    $headers = [
        'From: ' . $adminEmail,
        'Reply-To: ' . $adminEmail,
        'X-Mailer: PHP/' . phpversion(),
        'Content-Type: text/html; charset=UTF-8'
    ];
    
    return mail($adminEmail, $subject, $message, implode("\r\n", $headers));
}

logMessage("=== BACKUP VERIFICATION STARTED ===");

try {
    $backupDir = __DIR__ . '/../backups';
    $issues = [];
    
    // Create backup directory if it doesn't exist
    if (!is_dir($backupDir)) {
        mkdir($backupDir, 0755, true);
        logMessage("Created backup directory: $backupDir");
    }
    
    // Check if backup directory is writable
    if (!is_writable($backupDir)) {
        $issues[] = "Backup directory is not writable";
        logMessage("ERROR: Backup directory is not writable");
    } else {
        logMessage("Backup directory is writable");
    }
    
    // Check disk space for backups
    $diskFree = disk_free_space($backupDir);
    $diskFreeGB = round($diskFree / 1024 / 1024 / 1024, 2);
    
    if ($diskFreeGB < 1) {
        $issues[] = "Less than 1GB free space available for backups";
        logMessage("WARNING: Less than 1GB free space available");
    } else {
        logMessage("Available disk space: {$diskFreeGB}GB");
    }
    
    // Check for recent database backups
    $dbBackups = glob($backupDir . '/database_*.sql');
    $recentDbBackups = 0;
    $oldestDbBackup = null;
    $newestDbBackup = null;
    
    foreach ($dbBackups as $backup) {
        $fileTime = filemtime($backup);
        $fileAge = time() - $fileTime;
        
        if ($fileAge < 86400) { // 24 hours
            $recentDbBackups++;
        }
        
        if (!$oldestDbBackup || $fileTime < filemtime($oldestDbBackup)) {
            $oldestDbBackup = $backup;
        }
        
        if (!$newestDbBackup || $fileTime > filemtime($newestDbBackup)) {
            $newestDbBackup = $backup;
        }
    }
    
    if ($recentDbBackups == 0) {
        $issues[] = "No recent database backups found";
        logMessage("WARNING: No recent database backups found");
    } else {
        logMessage("Recent database backups: $recentDbBackups");
    }
    
    // Check for recent file backups
    $fileBackups = glob($backupDir . '/files_*.tar.gz');
    $recentFileBackups = 0;
    
    foreach ($fileBackups as $backup) {
        $fileTime = filemtime($backup);
        $fileAge = time() - $fileTime;
        
        if ($fileAge < 86400) { // 24 hours
            $recentFileBackups++;
        }
    }
    
    if ($recentFileBackups == 0) {
        $issues[] = "No recent file backups found";
        logMessage("WARNING: No recent file backups found");
    } else {
        logMessage("Recent file backups: $recentFileBackups");
    }
    
    // Test database backup creation
    try {
        $pdo = getCronDB();\n    if (!$pdo) {\n        throw new Exception("Database connection failed");\n    }
        $testBackupFile = $backupDir . '/test_backup_' . date('Y-m-d_H-i-s') . '.sql';
        
        // Create a simple test backup
        $tables = ['students', 'subjects', 'grades'];
        $backupContent = "-- Test backup created on " . date('Y-m-d H:i:s') . "\n";
        $backupContent .= "-- Database: " . DB_NAME . "\n\n";
        
        foreach ($tables as $table) {
            try {
                $stmt = $pdo->query("SHOW CREATE TABLE `$table`");
                $createTable = $stmt->fetch();
                if ($createTable) {
                    $backupContent .= $createTable['Create Table'] . ";\n\n";
                }
            } catch (Exception $e) {
                logMessage("WARNING: Could not backup table $table: " . $e->getMessage());
            }
        }
        
        if (file_put_contents($testBackupFile, $backupContent)) {
            logMessage("Database backup test successful");
            unlink($testBackupFile); // Clean up test file
        } else {
            $issues[] = "Database backup test failed";
            logMessage("ERROR: Database backup test failed");
        }
        
    } catch (Exception $e) {
        $issues[] = "Database connection failed: " . $e->getMessage();
        logMessage("ERROR: Database connection failed: " . $e->getMessage());
    }
    
    // Check backup file sizes
    $totalBackupSize = 0;
    $largeBackups = [];
    
    $allBackups = array_merge($dbBackups, $fileBackups);
    foreach ($allBackups as $backup) {
        $size = filesize($backup);
        $totalBackupSize += $size;
        
        if ($size > 1073741824) { // 1GB
            $largeBackups[] = basename($backup) . ' (' . round($size/1024/1024/1024, 2) . 'GB)';
        }
    }
    
    $totalBackupSizeGB = round($totalBackupSize / 1024 / 1024 / 1024, 2);
    logMessage("Total backup size: {$totalBackupSizeGB}GB");
    
    if (count($largeBackups) > 0) {
        logMessage("Large backup files: " . implode(', ', $largeBackups));
    }
    
    // Check for corrupted backups
    $corruptedBackups = 0;
    foreach ($allBackups as $backup) {
        if (filesize($backup) == 0) {
            $corruptedBackups++;
            $issues[] = "Corrupted backup file: " . basename($backup);
            logMessage("WARNING: Corrupted backup file: " . basename($backup));
        }
    }
    
    // Send alert if there are issues
    if (count($issues) > 0) {
        $alertSubject = "Backup Verification Alert - " . APP_NAME;
        $alertMessage = "
            <html>
            <body>
                <h2>Backup Verification Alert</h2>
                <p>The following issues were detected with your backup system:</p>
                <ul>
        ";
        
        foreach ($issues as $issue) {
            $alertMessage .= "<li>$issue</li>";
        }
        
        $alertMessage .= "
                </ul>
                <p>Please check your backup configuration and ensure backups are running properly.</p>
                <p>Server: " . ($_SERVER['SERVER_NAME'] ?? 'localhost') . "</p>
                <p>Time: " . date('Y-m-d H:i:s') . "</p>
            </body>
            </html>
        ";
        
        if (sendAlert($alertSubject, $alertMessage)) {
            logMessage("Alert email sent to administrator");
        } else {
            logMessage("WARNING: Failed to send alert email");
        }
    }
    
    logMessage("Backup verification completed:");
    logMessage("- Issues found: " . count($issues));
    logMessage("- Recent DB backups: $recentDbBackups");
    logMessage("- Recent file backups: $recentFileBackups");
    logMessage("- Corrupted backups: $corruptedBackups");
    logMessage("- Total backup size: {$totalBackupSizeGB}GB");
    
    if (count($issues) == 0) {
        logMessage("All backup checks passed successfully");
    }
    
} catch (Exception $e) {
    logMessage("ERROR: Backup verification failed: " . $e->getMessage());
    exit(1);
}

logMessage("=== BACKUP VERIFICATION COMPLETED ===");
?>
