-- =====================================================
-- TENANT FILE SYSTEM - DATABASE TABLES
-- =====================================================
-- Tables for logging file operations and access
-- =====================================================

-- =====================================================
-- 1. FILE OPERATIONS LOG
-- =====================================================
-- Logs all file operations (upload, delete, etc.)

CREATE TABLE IF NOT EXISTS file_operations_log (
    id BIGINT AUTO_INCREMENT PRIMARY KEY,
    tenant_id VARCHAR(50) NOT NULL,
    user_id INT NULL,
    operation VARCHAR(50) NOT NULL,
    message TEXT NOT NULL,
    metadata JSON NULL,
    ip_address VARCHAR(45) NULL,
    user_agent VARCHAR(255) NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    
    INDEX idx_tenant (tenant_id),
    INDEX idx_user (user_id),
    INDEX idx_operation (operation),
    INDEX idx_created (created_at)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- =====================================================
-- 2. FILE ACCESS LOG
-- =====================================================
-- Logs when users access/download files

CREATE TABLE IF NOT EXISTS file_access_log (
    id BIGINT AUTO_INCREMENT PRIMARY KEY,
    tenant_id VARCHAR(50) NOT NULL,
    user_id INT NOT NULL,
    file_type VARCHAR(50) NOT NULL,
    filename VARCHAR(255) NOT NULL,
    ip_address VARCHAR(45) NULL,
    user_agent VARCHAR(255) NULL,
    accessed_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    
    INDEX idx_tenant (tenant_id),
    INDEX idx_user (user_id),
    INDEX idx_file_type (file_type),
    INDEX idx_accessed (accessed_at),
    
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- =====================================================
-- 3. TENANT FILES REGISTRY (OPTIONAL)
-- =====================================================
-- Optional table to track all files in database
-- Useful for advanced features like file versioning, sharing, etc.

CREATE TABLE IF NOT EXISTS tenant_files (
    id BIGINT AUTO_INCREMENT PRIMARY KEY,
    tenant_id VARCHAR(50) NOT NULL,
    file_type VARCHAR(50) NOT NULL,
    filename VARCHAR(255) NOT NULL,
    original_name VARCHAR(255) NOT NULL,
    file_path VARCHAR(500) NOT NULL,
    relative_path VARCHAR(500) NOT NULL,
    file_size BIGINT NOT NULL,
    mime_type VARCHAR(100) NULL,
    extension VARCHAR(10) NULL,
    uploaded_by INT NULL,
    description TEXT NULL,
    is_public BOOLEAN DEFAULT FALSE,
    is_archived BOOLEAN DEFAULT FALSE,
    download_count INT DEFAULT 0,
    last_accessed_at TIMESTAMP NULL,
    uploaded_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    deleted_at TIMESTAMP NULL,
    
    INDEX idx_tenant (tenant_id),
    INDEX idx_type (file_type),
    INDEX idx_uploaded_by (uploaded_by),
    INDEX idx_public (is_public),
    INDEX idx_archived (is_archived),
    INDEX idx_uploaded (uploaded_at),
    
    FOREIGN KEY (uploaded_by) REFERENCES users(id) ON DELETE SET NULL
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- =====================================================
-- 4. FILE SHARES (OPTIONAL)
-- =====================================================
-- Optional table for sharing files between users

CREATE TABLE IF NOT EXISTS file_shares (
    id BIGINT AUTO_INCREMENT PRIMARY KEY,
    file_id BIGINT NOT NULL,
    tenant_id VARCHAR(50) NOT NULL,
    shared_by INT NOT NULL,
    shared_with INT NULL,
    share_token VARCHAR(64) NULL UNIQUE,
    permissions ENUM('view', 'download', 'edit') DEFAULT 'view',
    expires_at TIMESTAMP NULL,
    access_count INT DEFAULT 0,
    max_access_count INT NULL,
    is_active BOOLEAN DEFAULT TRUE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    last_accessed_at TIMESTAMP NULL,
    
    INDEX idx_file (file_id),
    INDEX idx_tenant (tenant_id),
    INDEX idx_token (share_token),
    INDEX idx_shared_with (shared_with),
    INDEX idx_active (is_active),
    
    FOREIGN KEY (file_id) REFERENCES tenant_files(id) ON DELETE CASCADE,
    FOREIGN KEY (shared_by) REFERENCES users(id) ON DELETE CASCADE,
    FOREIGN KEY (shared_with) REFERENCES users(id) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- =====================================================
-- 5. TENANT STORAGE QUOTAS
-- =====================================================
-- Track storage quotas and usage per tenant

CREATE TABLE IF NOT EXISTS tenant_storage_quotas (
    id INT AUTO_INCREMENT PRIMARY KEY,
    tenant_id VARCHAR(50) NOT NULL UNIQUE,
    storage_quota BIGINT DEFAULT 1073741824, -- 1GB default
    storage_used BIGINT DEFAULT 0,
    file_count INT DEFAULT 0,
    last_calculated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    quota_warning_sent BOOLEAN DEFAULT FALSE,
    quota_exceeded BOOLEAN DEFAULT FALSE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    
    INDEX idx_tenant (tenant_id),
    INDEX idx_quota_exceeded (quota_exceeded)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- =====================================================
-- INITIAL DATA
-- =====================================================

-- Create storage quota entries for existing tenants
INSERT IGNORE INTO tenant_storage_quotas (tenant_id, storage_quota)
SELECT DISTINCT academy_reference, 1073741824
FROM tenants
WHERE academy_reference IS NOT NULL;

-- =====================================================
-- USEFUL QUERIES
-- =====================================================

-- Get storage usage per tenant
-- SELECT 
--     tenant_id,
--     storage_used,
--     storage_quota,
--     ROUND((storage_used / storage_quota) * 100, 2) as usage_percentage,
--     file_count
-- FROM tenant_storage_quotas
-- ORDER BY usage_percentage DESC;

-- Get most accessed files
-- SELECT 
--     tf.tenant_id,
--     tf.filename,
--     tf.file_type,
--     COUNT(fal.id) as access_count,
--     MAX(fal.accessed_at) as last_accessed
-- FROM tenant_files tf
-- LEFT JOIN file_access_log fal ON tf.tenant_id = fal.tenant_id AND tf.filename = fal.filename
-- GROUP BY tf.id
-- ORDER BY access_count DESC
-- LIMIT 20;

-- Get file operations by user
-- SELECT 
--     u.username,
--     fol.operation,
--     COUNT(*) as operation_count,
--     MAX(fol.created_at) as last_operation
-- FROM file_operations_log fol
-- JOIN users u ON fol.user_id = u.id
-- WHERE fol.created_at >= DATE_SUB(NOW(), INTERVAL 30 DAY)
-- GROUP BY u.id, fol.operation
-- ORDER BY operation_count DESC;

-- =====================================================
-- MAINTENANCE QUERIES
-- =====================================================

-- Clean up old log entries (keep last 90 days)
-- DELETE FROM file_access_log WHERE accessed_at < DATE_SUB(NOW(), INTERVAL 90 DAY);
-- DELETE FROM file_operations_log WHERE created_at < DATE_SUB(NOW(), INTERVAL 90 DAY);

-- Recalculate storage usage for a tenant
-- UPDATE tenant_storage_quotas
-- SET storage_used = (
--     SELECT COALESCE(SUM(file_size), 0)
--     FROM tenant_files
--     WHERE tenant_id = tenant_storage_quotas.tenant_id
--     AND deleted_at IS NULL
-- ),
-- file_count = (
--     SELECT COUNT(*)
--     FROM tenant_files
--     WHERE tenant_id = tenant_storage_quotas.tenant_id
--     AND deleted_at IS NULL
-- ),
-- last_calculated_at = NOW()
-- WHERE tenant_id = 'YOUR_TENANT_ID';

-- =====================================================

