-- =====================================================
-- MOODLE INTEGRATION - SUBJECT TO COURSE MAPPING
-- =====================================================
-- This table maps internal subjects to Moodle courses
-- per school (academy_reference) to handle multi-tenant
-- =====================================================

-- Drop existing table if exists
DROP TABLE IF EXISTS subject_moodle_mapping;

-- Create mapping table
CREATE TABLE IF NOT EXISTS subject_moodle_mapping (
    id INT AUTO_INCREMENT PRIMARY KEY,
    
    -- Which school/tenant
    academy_reference VARCHAR(50) NOT NULL,
    
    -- Internal subject
    subject_id INT NOT NULL,
    
    -- Moodle course details
    moodle_course_id INT NULL COMMENT 'Moodle course ID',
    moodle_course_shortname VARCHAR(100) NULL COMMENT 'Moodle course shortname',
    moodle_course_idnumber VARCHAR(100) NULL COMMENT 'Moodle course idnumber',
    
    -- Enrollment settings
    auto_enroll TINYINT(1) DEFAULT 1 COMMENT 'Auto-enroll when subject assigned',
    enrollment_role VARCHAR(50) DEFAULT 'student' COMMENT 'Role in Moodle (student, teacher, etc)',
    
    -- Sync settings
    sync_enabled TINYINT(1) DEFAULT 1 COMMENT 'Enable sync for this mapping',
    last_sync_at DATETIME NULL COMMENT 'Last successful sync',
    sync_status VARCHAR(50) DEFAULT 'pending' COMMENT 'pending, success, failed',
    sync_error TEXT NULL COMMENT 'Last sync error message',
    
    -- Metadata
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    updated_at DATETIME DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    created_by INT NULL COMMENT 'Admin who created mapping',
    
    -- Indexes
    INDEX idx_academy (academy_reference),
    INDEX idx_subject (subject_id),
    INDEX idx_moodle_course (moodle_course_id),
    INDEX idx_auto_enroll (auto_enroll),
    INDEX idx_sync_enabled (sync_enabled),
    
    -- Unique constraint: one mapping per subject per school
    UNIQUE KEY unique_subject_per_school (academy_reference, subject_id),
    
    -- Foreign keys (commented out if tables don't exist yet)
    -- FOREIGN KEY (academy_reference) REFERENCES academy_references(reference_code) ON DELETE CASCADE,
    FOREIGN KEY (subject_id) REFERENCES subjects(id) ON DELETE CASCADE
    -- FOREIGN KEY (created_by) REFERENCES admins(id) ON DELETE SET NULL

) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
COMMENT='Maps internal subjects to Moodle courses for auto-enrollment';

-- =====================================================
-- ENROLLMENT LOG TABLE
-- =====================================================
-- Tracks all Moodle enrollment attempts
-- =====================================================

DROP TABLE IF EXISTS moodle_enrollment_log;

CREATE TABLE IF NOT EXISTS moodle_enrollment_log (
    id BIGINT AUTO_INCREMENT PRIMARY KEY,
    
    -- Student and course info
    student_id INT NOT NULL,
    subject_id INT NOT NULL,
    academy_reference VARCHAR(50) NOT NULL,
    
    -- Moodle details
    moodle_course_id INT NULL,
    moodle_user_id INT NULL,
    
    -- Enrollment details
    action VARCHAR(50) NOT NULL COMMENT 'enroll, unenroll, sync',
    status VARCHAR(50) NOT NULL COMMENT 'success, failed, pending',
    error_message TEXT NULL,
    
    -- Request/Response
    request_data TEXT NULL COMMENT 'JSON of request sent to Moodle',
    response_data TEXT NULL COMMENT 'JSON of Moodle response',
    
    -- Metadata
    attempted_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    completed_at DATETIME NULL,
    attempted_by INT NULL COMMENT 'Admin who triggered action',
    
    -- Indexes
    INDEX idx_student (student_id),
    INDEX idx_subject (subject_id),
    INDEX idx_academy (academy_reference),
    INDEX idx_status (status),
    INDEX idx_action (action),
    INDEX idx_attempted_at (attempted_at),
    
    FOREIGN KEY (student_id) REFERENCES students(id) ON DELETE CASCADE,
    FOREIGN KEY (subject_id) REFERENCES subjects(id) ON DELETE CASCADE
    -- FOREIGN KEY (academy_reference) REFERENCES academy_references(reference_code) ON DELETE CASCADE

) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
COMMENT='Logs all Moodle enrollment/unenrollment attempts';

-- =====================================================
-- MOODLE SYNC QUEUE
-- =====================================================
-- Queue for pending Moodle operations
-- =====================================================

DROP TABLE IF EXISTS moodle_sync_queue;

CREATE TABLE IF NOT EXISTS moodle_sync_queue (
    id BIGINT AUTO_INCREMENT PRIMARY KEY,
    
    -- Operation details
    operation_type VARCHAR(50) NOT NULL COMMENT 'enroll, unenroll, update_user, create_course',
    priority TINYINT DEFAULT 5 COMMENT '1-10, higher = more urgent',
    
    -- Target details
    student_id INT NULL,
    subject_id INT NULL,
    academy_reference VARCHAR(50) NOT NULL,
    
    -- Payload
    payload JSON NULL COMMENT 'Operation-specific data',
    
    -- Status
    status VARCHAR(50) DEFAULT 'pending' COMMENT 'pending, processing, completed, failed',
    attempts INT DEFAULT 0,
    max_attempts INT DEFAULT 3,
    error_message TEXT NULL,
    
    -- Timing
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    scheduled_at DATETIME DEFAULT CURRENT_TIMESTAMP COMMENT 'When to process',
    started_at DATETIME NULL,
    completed_at DATETIME NULL,
    next_retry_at DATETIME NULL,
    
    -- Indexes
    INDEX idx_status (status),
    INDEX idx_scheduled_at (scheduled_at),
    INDEX idx_priority (priority),
    INDEX idx_academy (academy_reference),
    INDEX idx_operation (operation_type),
    
    FOREIGN KEY (student_id) REFERENCES students(id) ON DELETE CASCADE,
    FOREIGN KEY (subject_id) REFERENCES subjects(id) ON DELETE CASCADE

) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
COMMENT='Queue for pending Moodle sync operations';

-- =====================================================
-- SAMPLE DATA (for testing)
-- =====================================================

-- Example: Map Grade 7 subjects to Moodle courses for KINE Academy
-- Uncomment and modify for your school

/*
INSERT INTO subject_moodle_mapping 
    (academy_reference, subject_id, moodle_course_id, moodle_course_shortname, moodle_course_idnumber, auto_enroll)
VALUES
    ('KINE', 1, 101, 'G7-MATH', 'G7-MAT-003', 1),  -- Mathematics
    ('KINE', 2, 102, 'G7-ENG', 'G7-ENG-001', 1),   -- English
    ('KINE', 3, 103, 'G7-SIS', 'G7-SIS-002', 1),   -- SiSwati
    ('KINE', 4, 104, 'G7-SCI', 'G7-SCI-004', 1);   -- Science
*/

-- =====================================================
-- VERIFICATION QUERIES
-- =====================================================

-- Check mapping table structure
DESCRIBE subject_moodle_mapping;

-- Check enrollment log
DESCRIBE moodle_enrollment_log;

-- Check sync queue
DESCRIBE moodle_sync_queue;

-- Count mappings per school
-- SELECT academy_reference, COUNT(*) as mapped_subjects 
-- FROM subject_moodle_mapping 
-- GROUP BY academy_reference;

-- Recent enrollment attempts
-- SELECT * FROM moodle_enrollment_log 
-- ORDER BY attempted_at DESC LIMIT 10;

-- Pending sync operations
-- SELECT * FROM moodle_sync_queue 
-- WHERE status = 'pending' 
-- ORDER BY priority DESC, scheduled_at ASC;

-- =====================================================
-- ROLLBACK SCRIPT
-- =====================================================

/*
DROP TABLE IF EXISTS moodle_sync_queue;
DROP TABLE IF EXISTS moodle_enrollment_log;
DROP TABLE IF EXISTS subject_moodle_mapping;
*/

-- =====================================================
-- COMPLETE!
-- =====================================================

