-- PAYMENT REMINDER SYSTEM - FINAL VERSION
-- Matches existing database collation

CREATE TABLE IF NOT EXISTS tenant_payment_settings (
    id INT AUTO_INCREMENT PRIMARY KEY,
    academy_reference VARCHAR(50) NOT NULL COLLATE utf8mb4_general_ci,
    payment_cycle_days INT NOT NULL DEFAULT 30,
    reminder_days_before INT NOT NULL DEFAULT 27,
    warning_days_after_reminder INT NOT NULL DEFAULT 3,
    days_before_suspension INT NOT NULL DEFAULT 35,
    monthly_fee DECIMAL(10,2) NOT NULL DEFAULT 0.00,
    unsuspension_fee DECIMAL(10,2) NOT NULL DEFAULT 0.00,
    late_payment_fee DECIMAL(10,2) NOT NULL DEFAULT 0.00,
    auto_suspend_enabled TINYINT(1) NOT NULL DEFAULT 1,
    auto_reminder_enabled TINYINT(1) NOT NULL DEFAULT 1,
    auto_warning_enabled TINYINT(1) NOT NULL DEFAULT 1,
    send_email_notifications TINYINT(1) NOT NULL DEFAULT 1,
    send_sms_notifications TINYINT(1) NOT NULL DEFAULT 1,
    send_parent_notifications TINYINT(1) NOT NULL DEFAULT 1,
    grace_period_days INT NOT NULL DEFAULT 0,
    allow_partial_payments TINYINT(1) NOT NULL DEFAULT 0,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    created_by INT NULL,
    UNIQUE KEY idx_academy_reference (academy_reference),
    INDEX idx_auto_suspend (auto_suspend_enabled)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci;

CREATE TABLE IF NOT EXISTS student_payment_schedules (
    id INT AUTO_INCREMENT PRIMARY KEY,
    student_id INT NOT NULL,
    academy_reference VARCHAR(50) NOT NULL COLLATE utf8mb4_general_ci,
    last_payment_date DATE NULL,
    next_due_date DATE NULL,
    suspension_date DATE NULL,
    payment_status ENUM('active', 'pending', 'overdue', 'suspended', 'grace_period') NOT NULL DEFAULT 'active',
    total_amount_due DECIMAL(10,2) NOT NULL DEFAULT 0.00,
    total_amount_paid DECIMAL(10,2) NOT NULL DEFAULT 0.00,
    balance DECIMAL(10,2) NOT NULL DEFAULT 0.00,
    reminder_sent_at TIMESTAMP NULL,
    reminder_count INT NOT NULL DEFAULT 0,
    warning_sent_at TIMESTAMP NULL,
    warning_count INT NOT NULL DEFAULT 0,
    last_notification_at TIMESTAMP NULL,
    suspended_at TIMESTAMP NULL,
    suspension_reason VARCHAR(255) NULL,
    unsuspended_at TIMESTAMP NULL,
    is_manually_suspended TINYINT(1) NOT NULL DEFAULT 0,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    UNIQUE KEY idx_student_academy (student_id, academy_reference),
    INDEX idx_student_id (student_id),
    INDEX idx_payment_status (payment_status),
    INDEX idx_next_due_date (next_due_date)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci;

CREATE TABLE IF NOT EXISTS payment_reminders (
    id INT AUTO_INCREMENT PRIMARY KEY,
    student_id INT NOT NULL,
    academy_reference VARCHAR(50) NOT NULL COLLATE utf8mb4_general_ci,
    schedule_id INT NOT NULL,
    reminder_type ENUM('initial_reminder', 'warning', 'final_warning', 'suspension_notice') NOT NULL,
    sent_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    due_date DATE NOT NULL,
    amount_due DECIMAL(10,2) NOT NULL,
    email_sent TINYINT(1) NOT NULL DEFAULT 0,
    email_sent_at TIMESTAMP NULL,
    email_error TEXT NULL,
    sms_sent TINYINT(1) NOT NULL DEFAULT 0,
    sms_sent_at TIMESTAMP NULL,
    sms_error TEXT NULL,
    parent_notified TINYINT(1) NOT NULL DEFAULT 0,
    parent_notified_at TIMESTAMP NULL,
    payment_link VARCHAR(500) NULL,
    payment_token VARCHAR(255) NULL,
    token_expires_at TIMESTAMP NULL,
    link_clicked TINYINT(1) NOT NULL DEFAULT 0,
    link_clicked_at TIMESTAMP NULL,
    payment_completed TINYINT(1) NOT NULL DEFAULT 0,
    payment_completed_at TIMESTAMP NULL,
    INDEX idx_student_id (student_id),
    INDEX idx_schedule_id (schedule_id),
    INDEX idx_reminder_type (reminder_type),
    INDEX idx_payment_token (payment_token)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci;

CREATE TABLE IF NOT EXISTS payment_transactions (
    id INT AUTO_INCREMENT PRIMARY KEY,
    student_id INT NOT NULL,
    academy_reference VARCHAR(50) NOT NULL COLLATE utf8mb4_general_ci,
    schedule_id INT NULL,
    reminder_id INT NULL,
    transaction_reference VARCHAR(100) NOT NULL UNIQUE,
    external_reference VARCHAR(100) NULL,
    amount DECIMAL(10,2) NOT NULL,
    currency VARCHAR(3) NOT NULL DEFAULT 'SZL',
    payment_method ENUM('momo', 'card', 'cash', 'bank_transfer', 'manual') NOT NULL,
    payment_provider VARCHAR(50) NULL,
    status ENUM('pending', 'processing', 'completed', 'failed', 'refunded', 'cancelled') NOT NULL DEFAULT 'pending',
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    paid_at TIMESTAMP NULL,
    failed_at TIMESTAMP NULL,
    payer_phone VARCHAR(20) NULL,
    payer_email VARCHAR(255) NULL,
    payer_name VARCHAR(255) NULL,
    raw_request TEXT NULL,
    raw_response TEXT NULL,
    error_message TEXT NULL,
    processed TINYINT(1) NOT NULL DEFAULT 0,
    processed_at TIMESTAMP NULL,
    processed_by INT NULL,
    description TEXT NULL,
    notes TEXT NULL,
    ip_address VARCHAR(45) NULL,
    user_agent VARCHAR(500) NULL,
    UNIQUE KEY idx_transaction_reference (transaction_reference),
    INDEX idx_student_id (student_id),
    INDEX idx_status (status)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci;

CREATE TABLE IF NOT EXISTS suspension_logs (
    id INT AUTO_INCREMENT PRIMARY KEY,
    student_id INT NOT NULL,
    academy_reference VARCHAR(50) NOT NULL COLLATE utf8mb4_general_ci,
    schedule_id INT NULL,
    suspended_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    unsuspended_at TIMESTAMP NULL,
    duration_days INT NULL,
    suspension_reason ENUM('payment_overdue', 'manual_admin', 'violation', 'other') NOT NULL DEFAULT 'payment_overdue',
    reason_description TEXT NULL,
    is_manual TINYINT(1) NOT NULL DEFAULT 0,
    suspended_by INT NULL,
    unsuspended_by INT NULL,
    admin_notes TEXT NULL,
    moodle_suspended TINYINT(1) NOT NULL DEFAULT 0,
    moodle_suspended_at TIMESTAMP NULL,
    moodle_unsuspended TINYINT(1) NOT NULL DEFAULT 0,
    moodle_unsuspended_at TIMESTAMP NULL,
    moodle_sync_error TEXT NULL,
    student_notified TINYINT(1) NOT NULL DEFAULT 0,
    parent_notified TINYINT(1) NOT NULL DEFAULT 0,
    notification_sent_at TIMESTAMP NULL,
    unsuspension_payment_required DECIMAL(10,2) NULL,
    unsuspension_payment_id INT NULL,
    status ENUM('active', 'lifted', 'expired') NOT NULL DEFAULT 'active',
    INDEX idx_student_id (student_id),
    INDEX idx_status (status)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci;

CREATE TABLE IF NOT EXISTS payment_notification_templates (
    id INT AUTO_INCREMENT PRIMARY KEY,
    academy_reference VARCHAR(50) NOT NULL COLLATE utf8mb4_general_ci,
    template_type ENUM('reminder', 'warning', 'suspension', 'payment_success', 'unsuspension') NOT NULL,
    notification_channel ENUM('email', 'sms', 'both') NOT NULL,
    subject VARCHAR(255) NULL,
    email_body TEXT NULL,
    sms_body TEXT NULL,
    is_active TINYINT(1) NOT NULL DEFAULT 1,
    is_default TINYINT(1) NOT NULL DEFAULT 0,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    created_by INT NULL,
    INDEX idx_academy_reference (academy_reference)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci;

CREATE TABLE IF NOT EXISTS payment_system_audit_log (
    id INT AUTO_INCREMENT PRIMARY KEY,
    user_id INT NULL,
    user_type ENUM('admin', 'student', 'parent', 'system') NOT NULL DEFAULT 'system',
    academy_reference VARCHAR(50) NOT NULL COLLATE utf8mb4_general_ci,
    action VARCHAR(100) NOT NULL,
    entity_type VARCHAR(50) NOT NULL,
    entity_id INT NULL,
    old_values JSON NULL,
    new_values JSON NULL,
    description TEXT NULL,
    ip_address VARCHAR(45) NULL,
    user_agent VARCHAR(500) NULL,
    request_url VARCHAR(500) NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    INDEX idx_academy_reference (academy_reference),
    INDEX idx_action (action)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci;

ALTER TABLE students 
ADD COLUMN IF NOT EXISTS suspended TINYINT(1) NOT NULL DEFAULT 0 AFTER email_verified,
ADD COLUMN IF NOT EXISTS suspended_at TIMESTAMP NULL AFTER suspended,
ADD COLUMN IF NOT EXISTS suspension_reason VARCHAR(255) NULL AFTER suspended_at;

INSERT INTO tenant_payment_settings (
    academy_reference, payment_cycle_days, reminder_days_before,
    warning_days_after_reminder, days_before_suspension,
    monthly_fee, unsuspension_fee, auto_suspend_enabled, auto_reminder_enabled
)
SELECT 
    reference_code COLLATE utf8mb4_general_ci, 30, 27, 3, 35, 350.00, 50.00, 1, 1
FROM academy_references
WHERE is_active = 1
  AND reference_code COLLATE utf8mb4_general_ci NOT IN (SELECT academy_reference FROM tenant_payment_settings);

DELIMITER //
DROP PROCEDURE IF EXISTS sp_init_student_payment_schedule//
CREATE PROCEDURE sp_init_student_payment_schedule(IN p_student_id INT, IN p_academy_reference VARCHAR(50) CHARSET utf8mb4 COLLATE utf8mb4_general_ci)
BEGIN
    DECLARE v_monthly_fee DECIMAL(10,2) DEFAULT 350.00;
    DECLARE v_payment_cycle_days INT DEFAULT 30;
    SELECT COALESCE(monthly_fee, 350.00), COALESCE(payment_cycle_days, 30)
    INTO v_monthly_fee, v_payment_cycle_days FROM tenant_payment_settings WHERE academy_reference = p_academy_reference LIMIT 1;
    INSERT INTO student_payment_schedules (student_id, academy_reference, last_payment_date, next_due_date, payment_status, total_amount_due, balance)
    VALUES (p_student_id, p_academy_reference, CURDATE(), DATE_ADD(CURDATE(), INTERVAL v_payment_cycle_days DAY), 'active', v_monthly_fee, v_monthly_fee)
    ON DUPLICATE KEY UPDATE last_payment_date = CURDATE(), next_due_date = DATE_ADD(CURDATE(), INTERVAL v_payment_cycle_days DAY);
END//

DROP PROCEDURE IF EXISTS sp_process_payment_success//
CREATE PROCEDURE sp_process_payment_success(IN p_transaction_id INT)
BEGIN
    DECLARE v_student_id INT; 
    DECLARE v_amount DECIMAL(10,2); 
    DECLARE v_schedule_id INT; 
    DECLARE v_academy_reference VARCHAR(50) CHARSET utf8mb4 COLLATE utf8mb4_general_ci; 
    DECLARE v_payment_cycle_days INT DEFAULT 30;
    SELECT student_id, amount, schedule_id, academy_reference INTO v_student_id, v_amount, v_schedule_id, v_academy_reference FROM payment_transactions WHERE id = p_transaction_id;
    SELECT COALESCE(payment_cycle_days, 30) INTO v_payment_cycle_days FROM tenant_payment_settings WHERE academy_reference = v_academy_reference LIMIT 1;
    UPDATE student_payment_schedules SET last_payment_date = CURDATE(), next_due_date = DATE_ADD(CURDATE(), INTERVAL v_payment_cycle_days DAY), payment_status = 'active', total_amount_paid = total_amount_paid + v_amount, balance = GREATEST(0, balance - v_amount), suspended_at = NULL, reminder_sent_at = NULL, warning_sent_at = NULL WHERE id = v_schedule_id;
    UPDATE suspension_logs SET unsuspended_at = CURRENT_TIMESTAMP, status = 'lifted', unsuspension_payment_id = p_transaction_id WHERE student_id = v_student_id AND status = 'active';
    UPDATE payment_transactions SET processed = 1, processed_at = CURRENT_TIMESTAMP WHERE id = p_transaction_id;
END//
DELIMITER ;

