<?php
/**
 * Quick Migration Script - Add Moodle Columns
 * This adds the missing moodle_status and related columns to fix the dashboard error
 */

require_once __DIR__ . '/../config.php';

echo "Adding Moodle and payment cycle columns to students table...\n\n";

try {
    // Get database connection
    $pdo = getDB();
    
    // Read the SQL file
    $sql = file_get_contents(__DIR__ . '/add_moodle_columns.sql');
    
    // Split into individual statements
    $statements = array_filter(array_map('trim', explode(';', $sql)));
    
    $successCount = 0;
    $skipCount = 0;
    $errorCount = 0;
    
    foreach ($statements as $statement) {
        if (empty($statement) || strpos($statement, '--') === 0) {
            continue;
        }
        
        try {
            $pdo->exec($statement);
            $successCount++;
            
            // Extract action description
            if (preg_match('/ALTER TABLE.*ADD COLUMN.*`(\w+)`/i', $statement, $matches)) {
                echo "✓ Added column: {$matches[1]}\n";
            } elseif (preg_match('/CREATE INDEX.*`(\w+)`/i', $statement, $matches)) {
                echo "✓ Created index: {$matches[1]}\n";
            } elseif (preg_match('/CREATE TABLE.*`(\w+)`/i', $statement, $matches)) {
                echo "✓ Created table: {$matches[1]}\n";
            } else {
                echo "✓ Executed statement successfully\n";
            }
            
        } catch (PDOException $e) {
            $errorMsg = $e->getMessage();
            
            // Check if it's a "already exists" error (which is fine)
            if (
                strpos($errorMsg, 'Duplicate column name') !== false ||
                strpos($errorMsg, 'Duplicate key name') !== false ||
                strpos($errorMsg, 'already exists') !== false ||
                strpos($errorMsg, "Table '") !== false && strpos($errorMsg, "' already exists") !== false
            ) {
                $skipCount++;
                if (preg_match('/`(\w+)`/', $statement, $matches)) {
                    echo "⚠ Already exists: {$matches[1]}\n";
                } else {
                    echo "⚠ Already exists (skipped)\n";
                }
            } else {
                $errorCount++;
                echo "✗ Error: " . $errorMsg . "\n";
            }
        }
    }
    
    echo "\n" . str_repeat("=", 60) . "\n";
    echo "Migration completed!\n";
    echo "  ✓ Success: $successCount\n";
    echo "  ⚠ Skipped (already exists): $skipCount\n";
    echo "  ✗ Errors: $errorCount\n";
    echo str_repeat("=", 60) . "\n\n";
    
    if ($errorCount === 0) {
        echo "✓ All migrations applied successfully!\n";
        echo "  The dashboard should now work without the 'moodle_status' error.\n\n";
    } else {
        echo "⚠ Some migrations failed. Please check the errors above.\n\n";
    }
    
} catch (Exception $e) {
    echo "✗ Fatal error: " . $e->getMessage() . "\n";
    exit(1);
}

