<?php
/**
 * Database migration script for soft deletion system
 * Run this script to add the required columns to the students table
 */

require_once __DIR__ . '/../config.php';

echo "=== DATABASE MIGRATION: SOFT DELETION SYSTEM ===\n\n";

try {
    $pdo = getDB();
    
    echo "✅ Database connection successful\n";
    
    // Check if columns already exist
    $stmt = $pdo->query("DESCRIBE students");
    $columns = $stmt->fetchAll(PDO::FETCH_COLUMN);
    
    $requiredColumns = ['deleted_at', 'permanent_delete_date', 'status'];
    $missingColumns = array_diff($requiredColumns, $columns);
    
    if (empty($missingColumns)) {
        echo "✅ All required columns already exist\n";
        echo "Migration not needed.\n";
        exit(0);
    }
    
    echo "📋 Missing columns: " . implode(', ', $missingColumns) . "\n\n";
    
    // Add missing columns
    if (in_array('deleted_at', $missingColumns)) {
        echo "➕ Adding deleted_at column...\n";
        $pdo->exec("ALTER TABLE students ADD COLUMN deleted_at DATETIME NULL");
        echo "✅ deleted_at column added\n";
    }
    
    if (in_array('permanent_delete_date', $missingColumns)) {
        echo "➕ Adding permanent_delete_date column...\n";
        $pdo->exec("ALTER TABLE students ADD COLUMN permanent_delete_date DATETIME NULL");
        echo "✅ permanent_delete_date column added\n";
    }
    
    if (in_array('status', $missingColumns)) {
        echo "➕ Adding status column...\n";
        $pdo->exec("ALTER TABLE students ADD COLUMN status ENUM('active', 'suspended') DEFAULT 'active'");
        echo "✅ status column added\n";
    }
    
    // Add indexes for better performance
    echo "\n📊 Adding performance indexes...\n";
    
    try {
        $pdo->exec("CREATE INDEX idx_students_status ON students(status)");
        echo "✅ Index on status column added\n";
    } catch (PDOException $e) {
        if (strpos($e->getMessage(), 'Duplicate key name') === false) {
            throw $e;
        }
        echo "ℹ️  Index on status column already exists\n";
    }
    
    try {
        $pdo->exec("CREATE INDEX idx_students_deleted_at ON students(deleted_at)");
        echo "✅ Index on deleted_at column added\n";
    } catch (PDOException $e) {
        if (strpos($e->getMessage(), 'Duplicate key name') === false) {
            throw $e;
        }
        echo "ℹ️  Index on deleted_at column already exists\n";
    }
    
    try {
        $pdo->exec("CREATE INDEX idx_students_permanent_delete_date ON students(permanent_delete_date)");
        echo "✅ Index on permanent_delete_date column added\n";
    } catch (PDOException $e) {
        if (strpos($e->getMessage(), 'Duplicate key name') === false) {
            throw $e;
        }
        echo "ℹ️  Index on permanent_delete_date column already exists\n";
    }
    
    // Update existing students to have 'active' status
    echo "\n🔄 Updating existing students...\n";
    $stmt = $pdo->prepare("UPDATE students SET status = 'active' WHERE status IS NULL");
    $stmt->execute();
    $updatedCount = $stmt->rowCount();
    echo "✅ Updated {$updatedCount} existing students to 'active' status\n";
    
    // Show final table structure
    echo "\n📋 Final table structure:\n";
    $stmt = $pdo->query("DESCRIBE students");
    $columns = $stmt->fetchAll();
    
    foreach ($columns as $column) {
        echo "- {$column['Field']}: {$column['Type']} " . 
             ($column['Null'] === 'YES' ? 'NULL' : 'NOT NULL') . 
             ($column['Default'] ? " DEFAULT {$column['Default']}" : '') . "\n";
    }
    
    echo "\n🎉 MIGRATION COMPLETED SUCCESSFULLY!\n";
    echo "The soft deletion system is now ready to use.\n";
    echo "\nNext steps:\n";
    echo "1. Upload the updated files to your cPanel\n";
    echo "2. Set up the cron job for automatic cleanup\n";
    echo "3. Test the soft deletion system\n";
    
} catch (Exception $e) {
    echo "❌ MIGRATION FAILED: " . $e->getMessage() . "\n";
    echo "\nPlease check your database connection and try again.\n";
    exit(1);
}
?>
