# Multi-Tenant System Testing Guide

## 🧪 Complete Testing Checklist

### Pre-Deployment Tests (Local Environment)

#### Test 1: Database Structure ✅
```bash
php tools/test_multi_tenant_isolation.php
```

**Expected Result:**
```
✓ students.academy_reference exists
✓ payments.academy_reference exists
✓ academy_references table exists
✓ tenant_settings table exists
✓ All students have academy_reference
✓ All indexes exist
```

---

#### Test 2: School Creation
**Steps:**
1. Login as super admin
2. Navigate to Super Admin → Add New School
3. Fill in details:
   - Name: "Test High School"
   - Code: "TEST"
   - Admin Email: test@test.com
   - Password: Test1234!
4. Click "Create School"

**Expected Result:**
- ✅ School created successfully
- ✅ Admin account created (test_admin)
- ✅ Wallet created with E0.00 balance
- ✅ Default settings populated
- ✅ School appears in school list

---

#### Test 3: School Isolation
**Steps:**
1. Create 2 students in School A (KINE)
2. Create 2 students in School B (TEST)
3. Login as School A admin
4. Verify only School A students visible
5. Try to access School B student via direct URL
6. Verify access denied

**Expected Result:**
- ✅ School A admin sees only School A students
- ✅ School B students not visible
- ✅ Direct URL access blocked
- ✅ No cross-school data leakage

---

#### Test 4: Payment Isolation
**Steps:**
1. Create payment for School A student
2. Create payment for School B student
3. Login as School A admin
4. Check wallet/payment dashboard

**Expected Result:**
- ✅ Only School A payments visible
- ✅ School A wallet shows only School A earnings
- ✅ Platform fees tracked separately per school

---

#### Test 5: Settings Isolation
**Steps:**
1. Set School A colors to Blue (#0000FF)
2. Set School B colors to Red (#FF0000)
3. Login to School A - verify blue theme
4. Login to School B - verify red theme

**Expected Result:**
- ✅ Each school has independent settings
- ✅ Colors don't cross over
- ✅ Settings persist correctly

---

### Security Tests

#### Test 6: Cross-School Data Access Attempts
```php
// Test script
<?php
require_once 'includes/functions.php';
require_once 'includes/school_context.php';

// Login as School A admin
setCurrentSchool('KINE');

// Try to query School B data
$pdo = getDB();
$stmt = $pdo->prepare("
    SELECT * FROM students 
    WHERE academy_reference = 'TEST'
");
$stmt->execute();
$results = $stmt->fetchAll();

// Should return data but admin shouldn't see it in UI
echo count($results) . " students found (raw query)\n";

// But with proper filtering:
$currentSchool = getCurrentSchool();
$stmt = $pdo->prepare("
    SELECT * FROM students 
    WHERE academy_reference = ?
");
$stmt->execute([$currentSchool]);
$filtered = $stmt->fetchAll();

echo count($filtered) . " students (filtered)\n";
```

**Expected Result:**
- ✅ Raw query works (database level)
- ✅ Filtered query returns only current school
- ✅ UI never shows cross-school data

---

#### Test 7: URL Manipulation
**Steps:**
1. Login as School A admin
2. Note a student ID from School A (e.g., ID=100)
3. Create student in School B (e.g., ID=101)
4. Try to access School B student:
   - `edit.php?id=101` from School A admin session

**Expected Result:**
- ✅ Access denied
- ✅ Redirected to dashboard
- ✅ Error message shown

---

#### Test 8: Session Hijacking Prevention
**Steps:**
1. Login as School A admin
2. Copy session ID
3. Manually change academy_reference in session to School B
4. Try to access dashboard

**Expected Result:**
- ✅ Session validation fails OR
- ✅ Data still filtered correctly

---

### Functionality Tests

#### Test 9: Student CRUD Operations
**For Each School:**
- [ ] Create student → Student has correct academy_reference
- [ ] View student → Can see student details
- [ ] Edit student → Changes save correctly
- [ ] Delete student → Soft delete works
- [ ] Restore student → Restore works

---

#### Test 10: Payment Processing
**For Each School:**
- [ ] Student registers
- [ ] Student makes payment (MTN MoMo)
- [ ] Payment tracked correctly
- [ ] Platform fee (E0.50) calculated
- [ ] School admin earnings calculated
- [ ] Wallet balance updated
- [ ] Payment shows in school's dashboard only

---

#### Test 11: Email System
**For Each School:**
- [ ] Email verification sent with school branding
- [ ] Welcome email uses school settings
- [ ] Email signature includes school name
- [ ] Emails sent from school email (if configured)

---

#### Test 12: Reporting
**For Each School:**
- [ ] Student list shows only their students
- [ ] Payment reports show only their payments
- [ ] Export CSV contains only their data
- [ ] Analytics filtered by school

---

### Performance Tests

#### Test 13: Query Performance
```sql
-- Test query speed with academy_reference filter
EXPLAIN SELECT * FROM students WHERE academy_reference = 'KINE';

-- Should use idx_students_academy_ref index
-- Expected: Using index, rows examined = actual rows
```

**Expected Result:**
- ✅ Query uses index
- ✅ Response time < 100ms
- ✅ No full table scans

---

#### Test 14: Concurrent Access
**Steps:**
1. Open 3 browser windows
2. Login to different schools in each
3. Perform actions simultaneously
4. Verify no data mixing

**Expected Result:**
- ✅ Each session independent
- ✅ No data confusion
- ✅ No performance degradation

---

### Integration Tests

#### Test 15: Moodle Sync (if applicable)
**For Each School:**
- [ ] Student created in system
- [ ] Bot syncs student to Moodle
- [ ] Student can login to Moodle
- [ ] Student enrolled in correct courses
- [ ] Sync status tracked per school

---

### User Acceptance Tests

#### Test 16: School Admin Workflow
**Simulate real usage:**
1. School admin logs in
2. Sees their school name in header
3. Creates 10 students
4. All students receive verification emails
5. Students verify emails
6. Students make payments
7. Admin sees earnings in wallet
8. Admin requests withdrawal

**Expected Result:**
- ✅ Smooth workflow
- ✅ No errors
- ✅ Data isolated correctly

---

## 📊 Test Results Template

```
Date: _______________
Tester: _______________
Environment: [ ] Local [ ] Staging [ ] Production

| Test # | Test Name | Status | Notes |
|--------|-----------|--------|-------|
| 1 | Database Structure | ✅ PASS | - |
| 2 | School Creation | ✅ PASS | - |
| 3 | School Isolation | ✅ PASS | - |
| 4 | Payment Isolation | ✅ PASS | - |
| 5 | Settings Isolation | ✅ PASS | - |
| 6 | Cross-School Access | ✅ PASS | - |
| 7 | URL Manipulation | ✅ PASS | - |
| 8 | Session Security | ✅ PASS | - |
| 9 | Student CRUD | ✅ PASS | - |
| 10 | Payment Processing | ✅ PASS | - |
| 11 | Email System | ✅ PASS | - |
| 12 | Reporting | ✅ PASS | - |
| 13 | Query Performance | ✅ PASS | - |
| 14 | Concurrent Access | ✅ PASS | - |
| 15 | Moodle Sync | ✅ PASS | - |
| 16 | User Acceptance | ✅ PASS | - |

OVERALL STATUS: [ ] PASS [ ] FAIL

Signature: _______________
```

---

## 🚀 Deployment Checklist

### Pre-Deployment
- [ ] All tests passed on local
- [ ] All tests passed on staging
- [ ] Database backup created
- [ ] Code backup created
- [ ] Rollback plan documented

### Deployment
- [ ] Put site in maintenance mode
- [ ] Run database migrations
- [ ] Deploy code changes
- [ ] Clear caches
- [ ] Test login for existing school (KINE)
- [ ] Create test school
- [ ] Verify isolation
- [ ] Remove maintenance mode

### Post-Deployment
- [ ] Monitor error logs
- [ ] Test all schools can access their dashboards
- [ ] Verify no cross-school access
- [ ] Check performance metrics
- [ ] Notify school admins of new features

---

## 📞 Support Escalation

### Level 1: School Admin Questions
- How to add students
- How to process payments
- How to view reports

**Response Time:** Within 24 hours  
**Contact:** info@melanegroup.com

### Level 2: Technical Issues
- Login problems
- Payment processing failures
- Email delivery issues

**Response Time:** Within 4 hours  
**Contact:** support@melanegroup.com

### Level 3: Critical Issues
- System down
- Data corruption
- Security breach

**Response Time:** Immediate  
**Contact:** +268 7687 3207 (Call/WhatsApp)

---

## ✅ Definition of Done

A school is successfully onboarded when:
- ✅ Admin can login
- ✅ Admin can create students
- ✅ Students receive verification emails
- ✅ Students can make payments
- ✅ Payments are tracked in school wallet
- ✅ Platform fees calculated correctly
- ✅ All data isolated from other schools
- ✅ School admin trained and satisfied
- ✅ No critical bugs reported in first week

---

**Average Onboarding Time:** 1-2 hours per school  
**Success Rate Target:** 100%  
**Support Required:** Low (system is self-service)

🎉 **Ready to onboard your first 20 schools!** 🎉

