<?php
/**
 * Moodle Suspension Handler
 * Handles suspension and unsuspension of users in Moodle via API
 */

require_once __DIR__ . '/../config.php';

class MoodleSuspensionHandler {
    
    private $pdo;
    private $moodle_url;
    private $moodle_token;
    
    public function __construct() {
        $this->pdo = getDB();
        $this->moodle_url = MOODLE_API_URL;
        $this->moodle_token = MOODLE_API_TOKEN;
    }
    
    /**
     * Suspend user in Moodle
     */
    public function suspendUser($student_id, $reason = 'payment_overdue') {
        try {
            $rawReason = trim((string)$reason);
            $allowedReasons = ['payment_overdue', 'manual_admin', 'violation', 'other'];
            $normalizedReason = strtolower(str_replace(' ', '_', $rawReason));
            if (!in_array($normalizedReason, $allowedReasons, true)) {
                $normalizedReason = 'other';
            }

            $reasonDescription = $rawReason !== '' ? $rawReason : ucfirst(str_replace('_', ' ', $normalizedReason));
            if (function_exists('mb_substr')) {
                $reasonDescription = mb_substr($reasonDescription, 0, 255, 'UTF-8');
            } else {
                $reasonDescription = substr($reasonDescription, 0, 255);
            }

            // Get student details
            $stmt = $this->pdo->prepare("
                SELECT id, username, email, moodle_user_id, academy_reference 
                FROM students 
                WHERE id = ?
            ");
            $stmt->execute([$student_id]);
            $student = $stmt->fetch();
            
            if (!$student) {
                throw new Exception("Student not found");
            }
            
            if (!$student['moodle_user_id']) {
                throw new Exception("Student not synced to Moodle");
            }
            
            // Suspend in Moodle (set suspended field to 1)
            $moodle_result = $this->callMoodleAPI('core_user_update_users', [
                'users' => [[
                    'id' => $student['moodle_user_id'],
                    'suspended' => 1
                ]]
            ]);
            
            if (isset($moodle_result['exception'])) {
                throw new Exception($moodle_result['message'] ?? 'Moodle API error');
            }
            
            // Log suspension
            $stmt = $this->pdo->prepare("
                INSERT INTO suspension_logs (
                    student_id, academy_reference, suspension_reason,
                    reason_description, moodle_suspended, moodle_suspended_at
                ) VALUES (?, ?, ?, ?, 1, NOW())
            ");
            $stmt->execute([
                $student_id,
                $student['academy_reference'],
                $normalizedReason,
                $reasonDescription
            ]);
            
            // Update student record
            $this->pdo->prepare("
                UPDATE students 
                SET suspended = 1, suspended_at = NOW(), suspension_reason = ?
                WHERE id = ?
            ")->execute([$reasonDescription, $student_id]);
            
            error_log("Student {$student_id} suspended in Moodle successfully");
            
            return [
                'success' => true,
                'message' => 'User suspended in Moodle',
                'moodle_user_id' => $student['moodle_user_id']
            ];
            
        } catch (Exception $e) {
            error_log("Failed to suspend user in Moodle: " . $e->getMessage());
            
            // Log failed suspension attempt
            if (isset($student)) {
                $this->pdo->prepare("
                    INSERT INTO suspension_logs (
                        student_id, academy_reference, suspension_reason,
                        moodle_suspended, moodle_sync_error
                    ) VALUES (?, ?, ?, 0, ?)
                ")->execute([
                    $student_id,
                    $student['academy_reference'],
                    $normalizedReason,
                    $e->getMessage()
                ]);
            }
            
            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }
    
    /**
     * Unsuspend user in Moodle
     */
    public function unsuspendUser($student_id) {
        try {
            // Get student details
            $stmt = $this->pdo->prepare("
                SELECT id, username, email, moodle_user_id, academy_reference 
                FROM students 
                WHERE id = ?
            ");
            $stmt->execute([$student_id]);
            $student = $stmt->fetch();
            
            if (!$student) {
                throw new Exception("Student not found");
            }
            
            if (!$student['moodle_user_id']) {
                throw new Exception("Student not synced to Moodle");
            }
            
            // Unsuspend in Moodle (set suspended field to 0)
            $moodle_result = $this->callMoodleAPI('core_user_update_users', [
                'users' => [[
                    'id' => $student['moodle_user_id'],
                    'suspended' => 0
                ]]
            ]);
            
            if (isset($moodle_result['exception'])) {
                throw new Exception($moodle_result['message'] ?? 'Moodle API error');
            }
            
            // Update suspension log
            $this->pdo->prepare("
                UPDATE suspension_logs 
                SET unsuspended_at = NOW(), status = 'lifted',
                    moodle_unsuspended = 1, moodle_unsuspended_at = NOW()
                WHERE student_id = ? AND status = 'active'
            ")->execute([$student_id]);
            
            // Update student record
            $this->pdo->prepare("
                UPDATE students 
                SET suspended = 0, suspended_at = NULL, suspension_reason = NULL
                WHERE id = ?
            ")->execute([$student_id]);
            
            error_log("Student {$student_id} unsuspended in Moodle successfully");
            
            return [
                'success' => true,
                'message' => 'User unsuspended in Moodle',
                'moodle_user_id' => $student['moodle_user_id']
            ];
            
        } catch (Exception $e) {
            error_log("Failed to unsuspend user in Moodle: " . $e->getMessage());
            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }
    
    /**
     * Get suspension status from Moodle
     */
    public function getSuspensionStatus($student_id) {
        try {
            $stmt = $this->pdo->prepare("
                SELECT moodle_user_id FROM students WHERE id = ?
            ");
            $stmt->execute([$student_id]);
            $student = $stmt->fetch();
            
            if (!$student || !$student['moodle_user_id']) {
                return ['success' => false, 'error' => 'Student not found in Moodle'];
            }
            
            // Get user info from Moodle
            $result = $this->callMoodleAPI('core_user_get_users_by_field', [
                'field' => 'id',
                'values' => [$student['moodle_user_id']]
            ]);
            
            if (empty($result) || isset($result['exception'])) {
                throw new Exception('Failed to get user from Moodle');
            }
            
            $user = $result[0];
            
            return [
                'success' => true,
                'suspended' => $user['suspended'] ?? 0,
                'moodle_user_id' => $student['moodle_user_id']
            ];
            
        } catch (Exception $e) {
            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }
    
    /**
     * Batch suspend multiple users
     */
    public function batchSuspend($student_ids, $reason = 'payment_overdue') {
        $results = [
            'success' => [],
            'failed' => []
        ];
        
        foreach ($student_ids as $student_id) {
            $result = $this->suspendUser($student_id, $reason);
            
            if ($result['success']) {
                $results['success'][] = $student_id;
            } else {
                $results['failed'][] = [
                    'student_id' => $student_id,
                    'error' => $result['error']
                ];
            }
        }
        
        return $results;
    }
    
    /**
     * Batch unsuspend multiple users
     */
    public function batchUnsuspend($student_ids) {
        $results = [
            'success' => [],
            'failed' => []
        ];
        
        foreach ($student_ids as $student_id) {
            $result = $this->unsuspendUser($student_id);
            
            if ($result['success']) {
                $results['success'][] = $student_id;
            } else {
                $results['failed'][] = [
                    'student_id' => $student_id,
                    'error' => $result['error']
                ];
            }
        }
        
        return $results;
    }
    
    /**
     * Call Moodle Web Service API
     */
    public function callMoodleAPI($function, $params = []) {
        $url = $this->moodle_url . '?wstoken=' . $this->moodle_token . 
               '&wsfunction=' . $function . '&moodlewsrestformat=json';
        
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, http_build_query($params));
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
        curl_setopt($ch, CURLOPT_TIMEOUT, 30);
        
        $response = curl_exec($ch);
        $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $curl_error = curl_error($ch);
        curl_close($ch);
        
        if ($curl_error) {
            throw new Exception("Curl error: " . $curl_error);
        }
        
        if ($http_code !== 200) {
            throw new Exception("HTTP error: " . $http_code);
        }
        
        $result = json_decode($response, true);
        
        if (json_last_error() !== JSON_ERROR_NONE) {
            throw new Exception("JSON decode error: " . json_last_error_msg());
        }
        
        return $result;
    }
}

