<?php
/**
 * School URL Generator
 * Generates clean URLs for school-specific pages
 */

class SchoolURLGenerator {
    
    private $base_url;
    private $school_code;
    
    public function __construct($school_code, $base_url = null) {
        $this->school_code = strtolower($school_code); // URLs are lowercase
        
        if ($base_url) {
            $this->base_url = rtrim($base_url, '/');
        } elseif (defined('PUBLIC_BASE_URL') && PUBLIC_BASE_URL) {
            $this->base_url = rtrim(PUBLIC_BASE_URL, '/');
        } else {
            // Auto-detect base URL
            $protocol = isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on' ? 'https' : 'http';
            $host = $_SERVER['HTTP_HOST'] ?? 'localhost';
            $this->base_url = $protocol . '://' . $host . '/Multi-Tanent';
        }
    }
    
    /**
     * Get all URLs for a school
     */
    public function getAllURLs() {
        return [
            'student_registration' => $this->getStudentRegistrationURL(),
            'parent_registration' => $this->getParentRegistrationURL(),
            'parent_login' => $this->getParentLoginURL(),
            'student_login' => $this->getStudentLoginURL(),
            'admin_login' => $this->getAdminLoginURL(),
            'school_info' => $this->getSchoolInfoURL(),
            'parent_dashboard' => $this->getParentDashboardURL()
        ];
    }
    
    /**
     * Student Registration URL
     * Clean: /school-code/register
     */
    public function getStudentRegistrationURL() {
        return $this->base_url . '/' . $this->school_code . '/register';
    }
    
    /**
     * Parent Registration URL
     * Clean: /school-code/parent/register
     */
    public function getParentRegistrationURL() {
        return $this->base_url . '/' . $this->school_code . '/parent/register';
    }
    
    /**
     * Parent Login URL
     * Clean: /school-code/parent/login
     */
    public function getParentLoginURL() {
        return $this->base_url . '/' . $this->school_code . '/parent/login';
    }
    
    /**
     * Student Login URL
     * Clean: /school-code/login
     */
    public function getStudentLoginURL() {
        return $this->base_url . '/' . $this->school_code . '/login';
    }
    
    /**
     * School Info URL
     * Clean: /school-code/info
     */
    public function getSchoolInfoURL() {
        return $this->base_url . '/' . $this->school_code . '/info';
    }
    
    /**
     * Parent Dashboard URL
     * Clean: /school-code/parent/dashboard
     */
    public function getParentDashboardURL() {
        return $this->base_url . '/' . $this->school_code . '/parent/dashboard';
    }

    /**
     * Admin Login URL
     * Clean: /admin/login.php?school=code
     */
    public function getAdminLoginURL() {
        return $this->base_url . '/admin/login.php?school=' . $this->school_code;
    }

    /**
     * Expose detected base URL
     */
    public function getBaseURL() {
        return $this->base_url;
    }
    
    /**
     * Generate QR Code data URL for a specific endpoint
     */
    public function getQRCodeURL($type = 'student_registration') {
        return null;
    }
    
    /**
     * Format URLs for display (with copy buttons)
     */
    public function getFormattedURLsHTML() {
        $urls = $this->getAllURLs();
        
        $html = '<div class="url-list">';
        
        foreach ($urls as $key => $url) {
            $label = $this->formatLabel($key);
            $html .= '
                <div class="mb-3">
                    <label class="form-label fw-bold">' . $label . '</label>
                    <div class="input-group">
                        <input type="text" class="form-control" value="' . htmlspecialchars($url) . '" readonly id="url_' . $key . '">
                        <button class="btn btn-outline-secondary" onclick="copyURL(\'url_' . $key . '\')">
                            <i class="bi bi-clipboard"></i> Copy
                        </button>
                        <a href="' . htmlspecialchars($url) . '" target="_blank" class="btn btn-outline-primary">
                            <i class="bi bi-box-arrow-up-right"></i> Open
                        </a>
                    </div>
                </div>
            ';
        }
        
        $html .= '</div>';
        
        return $html;
    }
    
    /**
     * Format label from key
     */
    private function formatLabel($key) {
        $labels = [
            'student_registration' => 'Student Registration',
            'parent_registration' => 'Parent Registration',
            'parent_login' => 'Parent Login',
            'student_login' => 'Student Login',
            'school_info' => 'School Information',
            'parent_dashboard' => 'Parent Dashboard'
        ];
        
        return $labels[$key] ?? ucwords(str_replace('_', ' ', $key));
    }
    
    /**
     * Generate shareable links HTML (for social media, etc.)
     */
    public function getShareableLinksHTML() {
        $studentRegURL = $this->getStudentRegistrationURL();
        $parentRegURL = $this->getParentRegistrationURL();
        
        $html = '
        <div class="share-links">
            <h6>Share Registration Links:</h6>
            <div class="btn-group" role="group">
                <a href="https://wa.me/?text=' . urlencode('Register for our school: ' . $studentRegURL) . '" 
                   target="_blank" class="btn btn-success btn-sm">
                    <i class="bi bi-whatsapp me-1"></i>WhatsApp
                </a>
                <a href="https://www.facebook.com/sharer/sharer.php?u=' . urlencode($studentRegURL) . '" 
                   target="_blank" class="btn btn-primary btn-sm">
                    <i class="bi bi-facebook me-1"></i>Facebook
                </a>
                <a href="https://twitter.com/intent/tweet?url=' . urlencode($studentRegURL) . '&text=' . urlencode('Student Registration') . '" 
                   target="_blank" class="btn btn-info btn-sm">
                    <i class="bi bi-twitter me-1"></i>Twitter
                </a>
                <a href="mailto:?subject=Student Registration&body=' . urlencode($studentRegURL) . '" 
                   class="btn btn-secondary btn-sm">
                    <i class="bi bi-envelope me-1"></i>Email
                </a>
            </div>
        </div>
        ';
        
        return $html;
    }
    
    /**
     * Get printable flyer content
     */
    public function getPrintableFlyerHTML($school_name) {
        $studentRegURL = $this->getStudentRegistrationURL();
        $qrCodeURL = $this->getQRCodeURL('student_registration');
        
        return '
        <div class="printable-flyer text-center p-5" style="border: 3px solid #0d6efd; border-radius: 10px;">
            <h2>' . htmlspecialchars($school_name) . '</h2>
            <h4 class="text-primary mb-4">Student Registration</h4>
            <img src="' . $qrCodeURL . '" alt="QR Code" class="mb-3" style="max-width: 250px;">
            <p class="lead">Scan QR Code or Visit:</p>
            <h5 class="mb-4"><code>' . htmlspecialchars($studentRegURL) . '</code></h5>
            <p class="text-muted">Register online in 10 minutes!</p>
        </div>
        ';
    }
}

