<?php
require_once __DIR__ . '/includes/functions.php';

$grades = getGrades();
$csrf = generateCSRFToken();
?>
<!doctype html>
<html lang="en">
<head>
	<meta charset="utf-8">
	<meta name="viewport" content="width=device-width, initial-scale=1">
	<title><?php echo htmlspecialchars(APP_NAME); ?> - Register</title>
	
	<!-- Favicon -->
	<link rel="icon" type="image/svg+xml" href="images/img-melanesys-logo.svg">
	<link rel="alternate icon" href="images/img-melanesys-logo.svg">
	<link rel="apple-touch-icon" href="images/img-melanesys-logo.svg">
	
	<link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
	<link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.3/font/bootstrap-icons.min.css" rel="stylesheet">
	<style>
		:root{
			--brand-primary:#6f42c1; /* purple */
			--brand-accent:#20c997;  /* teal */
			--brand-soft:#f3f0ff;    /* soft bg */
		}
		body{background:linear-gradient(135deg,var(--brand-soft),#e8fff9);} 
		.hero{
			background: radial-gradient(1000px 400px at 10% 0%, rgba(111,66,193,.15), rgba(32,201,151,.1)),
				linear-gradient(135deg, rgba(111,66,193,.12), rgba(32,201,151,.12));
			border-radius: 1rem;
		}
		.brand-badge{background:var(--brand-primary);}
		.brand-underline{position:relative; display:inline-block}
		.brand-underline::after{content:""; position:absolute; left:0; right:0; bottom:-.25rem; height:.35rem; background:linear-gradient(90deg,var(--brand-primary),var(--brand-accent)); border-radius:1rem}
		.card-modern{border:0; box-shadow:0 1rem 2rem rgba(0,0,0,.07);}
		.input-icon{position:relative}
		.input-icon .bi{position:absolute; left:.9rem; top:50%; transform:translateY(-50%); color:#6c757d}
		.input-icon input, .input-icon select{padding-left:2.4rem}
		.list-gradient .list-group-item{border:0; background:linear-gradient(90deg,#ffffff,#f8f9fa)}
		.btn-gradient{background:linear-gradient(90deg,var(--brand-primary),var(--brand-accent)); border:0}
		.btn-gradient:hover{filter:brightness(.95)}
		.form-check-input:checked{background-color:var(--brand-primary); border-color:var(--brand-primary)}
		.form-check-label{cursor:pointer; user-select:none}
		.subject-checkbox:hover{background-color:rgba(111,66,193,.05); border-radius:.375rem}
		
		/* Modern Form Enhancements */
		.form-section {
			background: rgba(248, 249, 250, 0.5);
			border-radius: 12px;
			padding: 1.5rem;
			border: 1px solid rgba(0, 0, 0, 0.05);
		}
		.section-title {
			font-size: 1rem;
			font-weight: 600;
			letter-spacing: 0.5px;
			text-transform: uppercase;
			border-bottom: 2px solid #e9ecef;
			padding-bottom: 0.5rem;
		}
		.form-control-lg {
			border-radius: 6px;
			border: 1.5px solid #e9ecef;
			transition: all 0.3s ease;
			font-size: 0.95rem;
			padding: 0.5rem 0.75rem;
		}
		.form-control-lg:focus {
			border-color: var(--brand-primary);
			box-shadow: 0 0 0 0.15rem rgba(111, 66, 193, 0.15);
		}
		.form-select-lg {
			border-radius: 6px;
			border: 1.5px solid #e9ecef;
			transition: all 0.3s ease;
			font-size: 0.95rem;
			padding: 0.5rem 0.75rem;
		}
		.form-select-lg:focus {
			border-color: var(--brand-primary);
			box-shadow: 0 0 0 0.15rem rgba(111, 66, 193, 0.15);
		}
		.form-label {
			color: #495057;
			margin-bottom: 0.4rem;
			font-size: 0.9rem;
			font-weight: 500;
		}
		.form-text {
			font-size: 0.8rem;
			color: #6c757d;
		}
		.section-title {
			font-size: 0.95rem !important;
			font-weight: 600;
			letter-spacing: 0.3px;
			text-transform: uppercase;
			border-bottom: 2px solid #e9ecef;
			padding-bottom: 0.4rem;
		}
		.form-section {
			background: rgba(248, 249, 250, 0.5);
			border-radius: 10px;
			padding: 1.2rem;
			border: 1px solid rgba(0, 0, 0, 0.05);
			margin-bottom: 1rem;
		}
		.alert-warning {
			background: rgba(255, 193, 7, 0.1);
			border: 1px solid rgba(255, 193, 7, 0.3);
			border-radius: 8px;
		}
		.form-actions {
			background: rgba(111, 66, 193, 0.05);
			border-radius: 12px;
			padding: 1.5rem;
			margin-top: 1rem;
		}
		.btn-lg {
			border-radius: 8px;
			font-weight: 600;
			letter-spacing: 0.5px;
		}
		#subjectsContainer {
			background: linear-gradient(135deg, #f8f9fa, #ffffff);
			border: 2px dashed #dee2e6;
			transition: all 0.3s ease;
		}
		#subjectsContainer:hover {
			border-color: var(--brand-primary);
		}
		.email-validation-status {
			z-index: 4;
		}
	</style>
</head>
<body>
	<div class="container py-4">
		<div class="hero p-4 p-md-5 mb-4 text-dark">
			<div class="d-flex align-items-center mb-2">
				<span class="badge brand-badge text-white me-2">New</span>
				<small class="text-muted">Simple. Fast. Secure.</small>
			</div>
			<h1 class="display-6 fw-bold mb-2 brand-underline"><?php echo htmlspecialchars(APP_NAME); ?> Registration</h1>
			<p class="mb-3">Create your student account, select your grade and subjects, and get started.</p>
			<div class="d-flex flex-column flex-sm-row gap-2">
				<a href="register_school.php" class="btn btn-primary btn-lg px-4">
					<i class="bi bi-building-add me-2"></i>Register Your School
				</a>
				<a href="select_school.php" class="btn btn-outline-secondary btn-lg px-4">
					<i class="bi bi-speedometer2 me-2"></i>School Admin Login
				</a>
			</div>
		</div>

		<div class="row g-4">
			<div class="col-lg-8">
				<div class="card card-modern">
					<div class="card-body p-4 p-md-5">
						<!-- Form Header -->
						<div class="text-center mb-4">
							<h4 class="text-primary mb-2">
								<i class="bi bi-person-plus-fill me-2"></i>
								Create Your Account
							</h4>
							<p class="text-muted mb-0">Join Skolo-Kine and start your learning journey</p>
						</div>

						<form method="post" action="register.php" novalidate>
							<input type="hidden" name="csrf_token" value="<?php echo htmlspecialchars($csrf); ?>">
							
							<!-- Personal Information Section -->
							<div class="form-section mb-4">
								<h6 class="section-title text-primary mb-3">
									<i class="bi bi-person-fill me-2"></i>Personal Information
								</h6>
								<div class="row g-3">
									<div class="col-md-6">
										<label class="form-label fw-semibold">First Name</label>
										<div class="input-icon">
											<i class="bi bi-person"></i>
											<input type="text" class="form-control form-control-lg" name="first_name" placeholder="e.g. Jane" required>
										</div>
									</div>
									<div class="col-md-6">
										<label class="form-label fw-semibold">Last Name</label>
										<div class="input-icon">
											<i class="bi bi-person"></i>
											<input type="text" class="form-control form-control-lg" name="last_name" placeholder="e.g. Doe" required>
										</div>
									</div>
								</div>
							</div>

							<!-- Contact Information Section -->
							<div class="form-section mb-4">
								<h6 class="section-title text-primary mb-3">
									<i class="bi bi-telephone-fill me-2"></i>Contact Information
								</h6>
								<div class="row g-3">
									<div class="col-md-6">
										<label class="form-label fw-semibold">Email Address</label>
										<div class="input-icon">
											<i class="bi bi-envelope"></i>
											<input type="email" class="form-control form-control-lg" name="email" placeholder="you@example.com" required>
											<div class="email-validation-status" style="position: absolute; right: 15px; top: 50%; transform: translateY(-50%); display: none;">
												<i class="bi bi-check-circle-fill text-success" style="font-size: 1.2rem;"></i>
											</div>
										</div>
										<div class="form-text">
											<i class="bi bi-info-circle me-1"></i>
											We'll verify this email address exists before registration.
										</div>
									</div>
									<div class="col-md-6">
										<label class="form-label fw-semibold">Phone Number</label>
										<div class="input-icon">
											<i class="bi bi-phone"></i>
											<input type="tel" class="form-control form-control-lg" name="phone" placeholder="e.g. 26876873207" required>
										</div>
										<div class="alert alert-info py-2 px-3 mt-2 mb-0" style="font-size: 0.8rem;">
											<i class="bi bi-info-circle-fill me-1"></i>
											<small>Use this for Mobile Money payments, or pay with card later.</small>
										</div>
									</div>
								</div>
							</div>

							<!-- Academic Information Section -->
							<div class="form-section mb-4">
								<h6 class="section-title text-primary mb-3">
									<i class="bi bi-mortarboard-fill me-2"></i>Academic Information
								</h6>
								<div class="row g-3">
									<div class="col-md-6">
										<label class="form-label fw-semibold">Grade Level</label>
										<div class="input-icon">
											<i class="bi bi-mortarboard"></i>
											<select class="form-select form-select-lg" name="grade_id" id="gradeSelect" required>
												<option value="">Select your grade...</option>
												<?php foreach ($grades as $g): ?>
													<option value="<?php echo (int)$g['id']; ?>"><?php echo htmlspecialchars($g['name']); ?></option>
												<?php endforeach; ?>
											</select>
										</div>
									</div>
									<div class="col-12">
										<label class="form-label fw-semibold">Select Subjects</label>
										<div id="subjectsContainer" class="border rounded-3 p-4 bg-light" style="min-height: 140px;">
											<div class="text-muted text-center py-5">
												<i class="bi bi-mortarboard fs-1 d-block mb-3 text-primary"></i>
												<h6 class="mb-2">Choose Your Subjects</h6>
												<p class="mb-0">Please select a grade first to see available subjects</p>
											</div>
										</div>
										<div class="form-text mt-2">
											<i class="bi bi-info-circle me-1"></i>
											Click on the subjects you want to select. You can choose multiple subjects.
										</div>
									</div>
								</div>
							</div>

							<!-- Security Section -->
							<div class="form-section mb-4">
								<h6 class="section-title text-primary mb-3">
									<i class="bi bi-shield-fill me-2"></i>Account Security
								</h6>
								<div class="row g-3">
									<div class="col-md-6">
										<label class="form-label fw-semibold">Create Password</label>
										<div class="input-icon">
											<i class="bi bi-shield-lock"></i>
											<input type="password" class="form-control form-control-lg" name="password" minlength="8" placeholder="At least 8 characters" required>
										</div>
										<div class="form-text">
											<i class="bi bi-lightbulb me-1"></i>
											Use a strong password you can remember.
										</div>
									</div>
									<div class="col-md-6">
										<label class="form-label fw-semibold">Confirm Password</label>
										<div class="input-icon">
											<i class="bi bi-shield-check"></i>
											<input type="password" class="form-control form-control-lg" name="password_confirm" minlength="8" placeholder="Re-type password" required>
										</div>
									</div>
								</div>
							</div>

							<!-- Form Actions -->
							<div class="form-actions text-center pt-3">
								<div class="d-grid d-sm-flex gap-3 justify-content-center">
									<button class="btn btn-gradient text-white px-5 py-3 btn-lg">
										<i class="bi bi-person-check me-2"></i>
										Create Account
									</button>
									<a class="btn btn-outline-secondary px-5 py-3 btn-lg" href="https://skoloi.melanegroup.com/kine" target="_blank">
										<i class="bi bi-box-arrow-in-right me-2"></i>
										Sign In Instead
									</a>
								</div>
								<div class="mt-3">
									<small class="text-muted">
										<i class="bi bi-lock-fill me-1"></i>
										Your information is secure and will only be used for educational purposes.
									</small>
								</div>
							</div>
						</form>
					</div>
				</div>
			</div>
			<div class="col-lg-4">
				<div class="card card-modern">
					<div class="card-body p-4">
						<h5 class="card-title mb-3">Subjects for Selected Grade</h5>
						<ul id="subjectsList" class="list-group list-gradient small"></ul>
						<div class="alert alert-info mt-3 mb-0">
							<i class="bi bi-info-circle me-1"></i> Subjects are configured by the admin and update dynamically per grade.
						</div>
					</div>
				</div>
			</div>
		</div>
	</div>
	<script>
	const gradeSelect = document.getElementById('gradeSelect');
	const subjectsList = document.getElementById('subjectsList');
	const subjectsContainer = document.getElementById('subjectsContainer');
	const form = document.querySelector('form');
	
	// Validation state
	let validationErrors = {};
	let emailValidationTimeout = null;
	
	function renderSubjects(items){
		// Update the preview list on the right
		subjectsList.innerHTML = '';
		if(!items || !items.length){
			const li = document.createElement('li');
			li.className = 'list-group-item';
			li.textContent = 'No subjects for this grade yet.';
			subjectsList.appendChild(li);
		} else {
			items.forEach(s => {
				const li = document.createElement('li');
				li.className = 'list-group-item d-flex justify-content-between align-items-center';
				li.textContent = s.name + (s.code ? ' ('+s.code+')' : '');
				subjectsList.appendChild(li);
			});
		}
		
		// Update the checkbox container
		subjectsContainer.innerHTML = '';
		if(!items || !items.length){
			subjectsContainer.innerHTML = `
				<div class="text-muted text-center py-4">
					<i class="bi bi-exclamation-circle fs-1 d-block mb-2"></i>
					No subjects available for this grade yet.
				</div>
			`;
			return;
		}
		
		// Add Select All / Clear All buttons
		const buttonRow = document.createElement('div');
		buttonRow.className = 'row mb-3';
		buttonRow.innerHTML = `
			<div class="col-12">
				<div class="d-flex gap-2">
					<button type="button" class="btn btn-sm btn-outline-primary" id="selectAllSubjects">
						<i class="bi bi-check-square me-1"></i>Select All
					</button>
					<button type="button" class="btn btn-sm btn-outline-secondary" id="clearAllSubjects">
						<i class="bi bi-square me-1"></i>Clear All
					</button>
				</div>
			</div>
		`;
		subjectsContainer.appendChild(buttonRow);
		
		// Create checkbox grid
		const row = document.createElement('div');
		row.className = 'row g-2';
		
		items.forEach(s => {
			const col = document.createElement('div');
			col.className = 'col-md-6 col-lg-4';
			
			const checkboxDiv = document.createElement('div');
			checkboxDiv.className = 'form-check subject-checkbox p-2';
			
			const checkbox = document.createElement('input');
			checkbox.type = 'checkbox';
			checkbox.className = 'form-check-input';
			checkbox.name = 'subject_ids[]';
			checkbox.value = s.id;
			checkbox.id = 'subject_' + s.id;
			
			const label = document.createElement('label');
			label.className = 'form-check-label w-100';
			label.htmlFor = 'subject_' + s.id;
			label.textContent = s.name + (s.code ? ' ('+s.code+')' : '');
			
			checkboxDiv.appendChild(checkbox);
			checkboxDiv.appendChild(label);
			col.appendChild(checkboxDiv);
			row.appendChild(col);
		});
		
		subjectsContainer.appendChild(row);
		
		// Add event listeners for Select All / Clear All buttons
		document.getElementById('selectAllSubjects').addEventListener('click', function() {
			const checkboxes = subjectsContainer.querySelectorAll('input[name="subject_ids[]"]');
			checkboxes.forEach(cb => cb.checked = true);
			validateField('subject_ids[]', '');
		});
		
		document.getElementById('clearAllSubjects').addEventListener('click', function() {
			const checkboxes = subjectsContainer.querySelectorAll('input[name="subject_ids[]"]');
			checkboxes.forEach(cb => cb.checked = false);
			validateField('subject_ids[]', '');
		});
	}
	
	async function loadSubjects(){
		const gid = gradeSelect.value;
		renderSubjects([]);
		if(!gid) return;
		
		// Check if it's Grade 7 (assuming Grade 7 has ID 1 or we can check by name)
		const selectedGradeText = gradeSelect.options[gradeSelect.selectedIndex].text;
		
		if(selectedGradeText !== 'Grade 7') {
			// Show unavailable message for non-Grade 7
			subjectsContainer.innerHTML = `
				<div class="text-center py-5">
					<i class="bi bi-exclamation-triangle text-warning fs-1 d-block mb-3"></i>
					<h5 class="text-warning">Grade Not Available Yet</h5>
					<p class="text-muted mb-0">Registration for ${selectedGradeText} is not available at the moment.</p>
					<p class="text-muted">Please select Grade 7 to continue with registration.</p>
				</div>
			`;
			return;
		}
		
		try{
			const res = await fetch('register.php?action=subjects&grade_id='+encodeURIComponent(gid), {headers: {'Accept':'application/json'}});
			if(!res.ok) return;
			const data = await res.json();
			renderSubjects(data || []);
		}catch(e){/* noop */}
	}
	
	// Validation functions
	function showFieldError(fieldName, message) {
		if (fieldName === 'subject_ids[]') {
			// Special handling for subjects checkboxes
			subjectsContainer.classList.add('border-danger');
			subjectsContainer.classList.remove('border-success');
			
			// Remove existing error message
			const existingError = subjectsContainer.querySelector('.invalid-feedback');
			if (existingError) {
				existingError.remove();
			}
			
			// Add error message
			const errorDiv = document.createElement('div');
			errorDiv.className = 'invalid-feedback d-block text-danger mt-2';
			errorDiv.textContent = message;
			subjectsContainer.appendChild(errorDiv);
			
			validationErrors[fieldName] = message;
			return;
		}
		
		const field = document.querySelector(`[name="${fieldName}"]`);
		if (!field) return;
		
		// Remove existing error styling
		field.classList.remove('is-invalid');
		field.classList.add('is-valid');
		
		// Remove existing error message
		const existingError = field.parentNode.querySelector('.invalid-feedback');
		if (existingError) {
			existingError.remove();
		}
		
		// Add error styling and message
		field.classList.remove('is-valid');
		field.classList.add('is-invalid');
		
		const errorDiv = document.createElement('div');
		errorDiv.className = 'invalid-feedback';
		errorDiv.textContent = message;
		field.parentNode.appendChild(errorDiv);
		
		validationErrors[fieldName] = message;
	}
	
	function clearFieldError(fieldName) {
		if (fieldName === 'subject_ids[]') {
			// Special handling for subjects checkboxes
			subjectsContainer.classList.remove('border-danger');
			subjectsContainer.classList.add('border-success');
			
			const existingError = subjectsContainer.querySelector('.invalid-feedback');
			if (existingError) {
				existingError.remove();
			}
			
			delete validationErrors[fieldName];
			return;
		}
		
		const field = document.querySelector(`[name="${fieldName}"]`);
		if (!field) return;
		
		field.classList.remove('is-invalid');
		field.classList.add('is-valid');
		
		const existingError = field.parentNode.querySelector('.invalid-feedback');
		if (existingError) {
			existingError.remove();
		}
		
		// Special handling for email field - hide status indicator when clearing errors
		if (fieldName === 'email') {
			const statusIndicator = document.querySelector('.email-validation-status');
			if (statusIndicator) {
				statusIndicator.style.display = 'none';
			}
		}
		
		delete validationErrors[fieldName];
	}
	
	function validateField(fieldName, value) {
		switch(fieldName) {
			case 'first_name':
			case 'last_name':
				if (!value || value.trim().length < 2) {
					showFieldError(fieldName, 'This field is required and must be at least 2 characters long.');
					return false;
				}
				clearFieldError(fieldName);
				return true;
				
			case 'email':
				const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
				if (!value || !emailRegex.test(value)) {
					showFieldError(fieldName, 'Please enter a valid email address.');
					return false;
				}
				// Note: Real-time email validation is handled separately via AJAX
				clearFieldError(fieldName);
				return true;
				
			case 'phone':
				// Validate phone number format (Swaziland format: 268XXXXXXXX)
				const phoneRegex = /^268[0-9]{8}$/;
				if (!value || !phoneRegex.test(value.replace(/\s+/g, ''))) {
					showFieldError(fieldName, 'Please enter a valid Swaziland phone number (format: 268XXXXXXXX, e.g., 26876873207).');
					return false;
				}
				clearFieldError(fieldName);
				return true;
				
			case 'grade_id':
				if (!value || value === '') {
					showFieldError(fieldName, 'Please select a grade.');
					return false;
				}
				// Check if Grade 7 is selected
				const selectedGradeText = gradeSelect.options[gradeSelect.selectedIndex].text;
				if (selectedGradeText !== 'Grade 7') {
					showFieldError(fieldName, 'Only Grade 7 registration is available at the moment.');
					return false;
				}
				clearFieldError(fieldName);
				return true;
				
			case 'password':
				if (!value || value.length < 8) {
					showFieldError(fieldName, 'Password must be at least 8 characters long.');
					return false;
				}
				clearFieldError(fieldName);
				return true;
				
			case 'password_confirm':
				const password = document.querySelector('[name="password"]').value;
				if (!value || value !== password) {
					showFieldError(fieldName, 'Passwords do not match.');
					return false;
				}
				clearFieldError(fieldName);
				return true;
				
			case 'subject_ids[]':
				const selectedCheckboxes = document.querySelectorAll('input[name="subject_ids[]"]:checked');
				if (selectedCheckboxes.length === 0) {
					showFieldError('subject_ids[]', 'Please select at least one subject.');
					return false;
				}
				clearFieldError('subject_ids[]');
				return true;
		}
		return true;
	}
	
	function validateForm() {
		validationErrors = {};
		let isValid = true;
		
		// Validate all required fields
		const requiredFields = ['first_name', 'last_name', 'email', 'phone', 'grade_id', 'password', 'password_confirm'];
		
		requiredFields.forEach(fieldName => {
			const field = document.querySelector(`[name="${fieldName}"]`);
			if (field) {
				if (!validateField(fieldName, field.value)) {
					isValid = false;
				}
			}
		});
		
		// Validate subjects selection
		if (!validateField('subject_ids[]', '')) {
			isValid = false;
		}
		
		return isValid;
	}
	
	// Real-time email validation function
	async function validateEmailRealTime(email) {
		if (!email || email.length < 5) {
			return; // Don't validate very short inputs
		}
		
		const emailField = document.querySelector('[name="email"]');
		const statusIndicator = document.querySelector('.email-validation-status');
		
		// Show loading state
		statusIndicator.innerHTML = '<i class="bi bi-hourglass-split text-warning" style="font-size: 1.2rem;"></i>';
		statusIndicator.style.display = 'block';
		
		try {
			const response = await fetch(`register.php?action=validate_email&email=${encodeURIComponent(email)}`);
			const result = await response.json();
			
			if (result.valid) {
				clearFieldError('email');
				// Show success indicator
				emailField.classList.remove('is-invalid');
				emailField.classList.add('is-valid');
				statusIndicator.innerHTML = '<i class="bi bi-check-circle-fill text-success" style="font-size: 1.2rem;"></i>';
			} else {
				showFieldError('email', result.message);
				statusIndicator.style.display = 'none';
			}
		} catch (error) {
			console.error('Email validation error:', error);
			statusIndicator.style.display = 'none';
			// Don't show error for network issues, just log them
		}
	}
	
	// Add real-time validation
	document.querySelectorAll('input, select').forEach(field => {
		field.addEventListener('blur', function() {
			validateField(this.name, this.value);
		});
		
		field.addEventListener('input', function() {
			// Clear error on input for better UX
			if (this.classList.contains('is-invalid')) {
				clearFieldError(this.name);
			}
			
			// Special handling for email field
			if (this.name === 'email') {
				// Clear previous timeout
				if (emailValidationTimeout) {
					clearTimeout(emailValidationTimeout);
				}
				
				// Set new timeout for email validation (debounce)
				emailValidationTimeout = setTimeout(() => {
					validateEmailRealTime(this.value);
				}, 1000); // Wait 1 second after user stops typing
			}
		});
	});
	
	// Special handling for password confirmation
	document.querySelector('[name="password"]').addEventListener('input', function() {
		const confirmField = document.querySelector('[name="password_confirm"]');
		if (confirmField.value) {
			validateField('password_confirm', confirmField.value);
		}
	});
	
	// Handle subjects selection validation (using event delegation for dynamically created checkboxes)
	subjectsContainer.addEventListener('change', function(e) {
		if (e.target.name === 'subject_ids[]') {
			validateField('subject_ids[]', '');
		}
	});
	
	// Form submission validation
	form.addEventListener('submit', function(e) {
		if (!validateForm()) {
			e.preventDefault();
			
			// Show general error message
			let errorMessage = 'Please fix the following errors:\n';
			Object.values(validationErrors).forEach(error => {
				errorMessage += '• ' + error + '\n';
			});
			
			alert(errorMessage);
			return false;
		}
	});
	
	gradeSelect.addEventListener('change', loadSubjects);
	</script>
</body>
</html>