<?php
require_once '../../config.php';
require_once '../../includes/functions.php';

header('Content-Type: application/json');

// Validate CSRF token
if (!validateCSRFToken($_POST['csrf_token'] ?? '')) {
    http_response_code(403);
    echo json_encode(['error' => 'Invalid security token']);
    exit;
}

// Get reference_id from request
$referenceId = trim($_POST['reference_id'] ?? '');
if (empty($referenceId)) {
    http_response_code(400);
    echo json_encode(['error' => 'reference_id is required']);
    exit;
}

try {
    $pdo = getDB();
    
    // Get payment record
    $stmt = $pdo->prepare("SELECT * FROM payments WHERE reference_id = ?");
    $stmt->execute([$referenceId]);
    $payment = $stmt->fetch();
    
    if (!$payment) {
        http_response_code(404);
        echo json_encode(['error' => 'Payment not found']);
        exit;
    }
    
    // If already completed or failed, return current status
    if (in_array($payment['status'], ['SUCCESSFUL', 'FAILED', 'CANCELLED'])) {
        echo json_encode([
            'success' => true,
            'reference_id' => $referenceId,
            'external_id' => $payment['external_id'],
            'status' => $payment['status'],
            'amount' => $payment['amount'],
            'currency' => $payment['currency'],
            'message' => 'Payment ' . strtolower($payment['status'])
        ]);
        exit;
    }
    
    // Check payment status with MTN MoMo API
    $api_url = MOMO_COLLECTION_BASE_URL . '/v1_0/requesttopay/' . $referenceId;
    $accessToken = getMomoAccessToken();
    $headers = [
        'Authorization: Bearer ' . $accessToken,
        'X-Target-Environment: ' . MOMO_ENV,
        'Ocp-Apim-Subscription-Key: ' . MOMO_COLLECTION_SUBSCRIPTION_KEY
    ];
    
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $api_url);
    curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_TIMEOUT, 10);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, true);
    
    $response = curl_exec($ch);
    $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $error = curl_error($ch);
    curl_close($ch);
    
    if ($error) {
        throw new Exception("CURL Error: $error");
    }
    
    if ($http_code == 200) {
        $data = json_decode($response, true);
        $status = $data['status'] ?? 'UNKNOWN';
        
        // Map MTN MoMo status to our status
        $status_map = [
            'SUCCESSFUL' => 'SUCCESSFUL',
            'FAILED' => 'FAILED',
            'PENDING' => 'PENDING',
            'CANCELLED' => 'CANCELLED'
        ];
        
        $new_status = $status_map[$status] ?? 'PENDING';
        
        // Update payment status
        $stmt = $pdo->prepare("
            UPDATE payments 
            SET status = ?, 
                updated_at = NOW(),
                momo_status = ?,
                raw_response = ?
            WHERE reference_id = ?
        ");
        $stmt->execute([$new_status, $status, $response, $referenceId]);
        
        // If payment is successful, update student status
        if ($new_status === 'SUCCESSFUL') {
            $stmt = $pdo->prepare("
                UPDATE students 
                SET payment_status = 'PAID', 
                    payment_date = NOW() 
                WHERE id = ?
            ");
            $stmt->execute([$payment['student_id']]);
        }
        
        echo json_encode([
            'success' => true,
            'reference_id' => $referenceId,
            'external_id' => $payment['external_id'],
            'status' => $new_status,
            'amount' => $payment['amount'],
            'currency' => $payment['currency'],
            'message' => 'Payment ' . strtolower($new_status)
        ]);
        
    } else {
        // API error
        $error_data = json_decode($response, true);
        $error_message = $error_data['message'] ?? 'Failed to check payment status';
        
        echo json_encode([
            'success' => false,
            'reference_id' => $referenceId,
            'external_id' => $payment['external_id'],
            'status' => 'ERROR',
            'message' => $error_message
        ]);
    }
    
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode(['error' => 'Internal server error: ' . $e->getMessage()]);
}

// Function to get MTN MoMo access token
function getMomoAccessToken() {
    $token_url = MOMO_COLLECTION_BASE_URL . '/token/';
    $headers = [
        'Authorization: Basic ' . base64_encode(MOMO_API_USER_ID . ':' . MOMO_API_KEY),
        'Ocp-Apim-Subscription-Key: ' . MOMO_COLLECTION_SUBSCRIPTION_KEY
    ];
    
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $token_url);
    curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_TIMEOUT, 10);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, true);
    curl_setopt($ch, CURLOPT_POST, true);
    
    $response = curl_exec($ch);
    $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    
    if ($http_code == 200) {
        $data = json_decode($response, true);
        if (!empty($data['access_token'])) {
            return $data['access_token'];
        }
    }
    
    throw new Exception('Failed to get access token');
}
?>
