<?php
/**
 * Process Payment from Reminder Link
 * Handles both MTN MoMo and Card payments
 */

session_start();
require_once 'config.php';
require_once 'includes/functions.php';
require_once 'includes/payment_reminder_manager.php';
require_once 'includes/url_helper.php';

$pdo = getDB();

// Validate form submission
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    header('Location: ' . generateURL('index.php'));
    exit;
}

// Get form data
$token = $_POST['token'] ?? '';
$student_id = (int)($_POST['student_id'] ?? 0);
$amount = (float)($_POST['amount'] ?? 0);
$schedule_id = (int)($_POST['schedule_id'] ?? 0);
$academy_reference = $_POST['academy_reference'] ?? '';
$payment_method = $_POST['payment_method'] ?? '';
$momo_phone = $_POST['momo_phone'] ?? '';

// Validate token
$manager = new PaymentReminderManager();
$token_result = $manager->decryptToken($token);

if (!$token_result['success']) {
    $_SESSION['error'] = 'Invalid or expired payment link';
    header('Location: ' . generateURL('pay_reminder.php') . '?token=' . urlencode($token));
    exit;
}

// Validate data
if (empty($student_id) || empty($amount) || empty($payment_method)) {
    $_SESSION['error'] = 'Invalid payment data';
    header('Location: ' . generateURL('pay_reminder.php') . '?token=' . urlencode($token));
    exit;
}

// Get student details
$stmt = $pdo->prepare("
    SELECT s.*, sps.*, ar.academy_name 
    FROM students s
    INNER JOIN student_payment_schedules sps ON s.id = sps.student_id
    INNER JOIN academy_references ar ON s.academy_reference = ar.reference_code
    WHERE s.id = ?
");
$stmt->execute([$student_id]);
$student = $stmt->fetch();

if (!$student) {
    $_SESSION['error'] = 'Student not found';
    header('Location: ' . generateURL('pay_reminder.php') . '?token=' . urlencode($token));
    exit;
}

// Generate transaction reference
$transaction_reference = 'REM-' . strtoupper(uniqid());

try {
    // Create payment transaction record
    $stmt = $pdo->prepare("
        INSERT INTO payment_transactions (
            student_id, academy_reference, schedule_id,
            transaction_reference, amount, currency,
            payment_method, status, payer_phone, payer_email, payer_name,
            description, ip_address, user_agent
        ) VALUES (?, ?, ?, ?, ?, ?, ?, 'pending', ?, ?, ?, ?, ?, ?)
    ");
    
    $stmt->execute([
        $student_id,
        $academy_reference,
        $schedule_id,
        $transaction_reference,
        $amount,
        'SZL',
        $payment_method,
        $momo_phone ? '+268' . $momo_phone : $student['phone'],
        $student['email'],
        $student['full_name'],
        'Payment from reminder link - ' . date('Y-m-d H:i:s'),
        $_SERVER['REMOTE_ADDR'] ?? null,
        $_SERVER['HTTP_USER_AGENT'] ?? null
    ]);
    
    $transaction_id = $pdo->lastInsertId();
    
    // Store transaction ID in session
    $_SESSION['transaction_id'] = $transaction_id;
    $_SESSION['student_id'] = $student_id;
    $_SESSION['academy_reference'] = $academy_reference;
    
    // Process payment based on method
    if ($payment_method === 'momo') {
        // MTN Mobile Money Payment
        $phone = '+268' . $momo_phone;
        
        // Call MTN MoMo API
        require_once 'includes/functions.php';
        $momo_result = initiateMTNMoMoPayment([
            'amount' => $amount,
            'currency' => 'EUR', // MTN sandbox uses EUR
            'phone' => $phone,
            'reference' => $transaction_reference,
            'payer_message' => 'School fee payment',
            'payee_note' => 'Payment for ' . $student['full_name']
        ]);
        
        if ($momo_result['success']) {
            // Update transaction with external reference
            $pdo->prepare("
                UPDATE payment_transactions 
                SET external_reference = ?, status = 'processing', raw_request = ?
                WHERE id = ?
            ")->execute([
                $momo_result['reference_id'],
                json_encode($momo_result),
                $transaction_id
            ]);
            
            // Redirect to status page
            $_SESSION['success'] = 'Payment request sent! Check your phone for the MoMo payment prompt.';
            header('Location: ' . generateURL('payment_status_reminder.php') . '?ref=' . urlencode($transaction_reference));
            exit;
            
        } else {
            throw new Exception($momo_result['error'] ?? 'Failed to initiate MoMo payment');
        }
        
    } else if ($payment_method === 'card') {
        // Card Payment - Redirect to card gateway
        
        // Store payment details in session
        $_SESSION['pending_card_payment'] = [
            'transaction_id' => $transaction_id,
            'amount' => $amount,
            'student_name' => $student['full_name'],
            'email' => $student['email'],
            'reference' => $transaction_reference
        ];
        
        // Redirect to card payment gateway
        header('Location: ' . generateURL('payments/card/request.php'));
        exit;
        
    } else {
        throw new Exception('Invalid payment method');
    }
    
} catch (Exception $e) {
    error_log("Payment processing error: " . $e->getMessage());
    
    // Update transaction as failed
    if (isset($transaction_id)) {
        $pdo->prepare("
            UPDATE payment_transactions 
            SET status = 'failed', error_message = ?, failed_at = NOW()
            WHERE id = ?
        ")->execute([$e->getMessage(), $transaction_id]);
    }
    
    $_SESSION['error'] = 'Payment failed: ' . $e->getMessage();
    header('Location: ' . generateURL('pay_reminder.php') . '?token=' . urlencode($token));
    exit;
}

