<?php
/**
 * List All School Admins
 * Super Admin views and manages all school admin accounts
 */

require_once '../../includes/functions.php';
require_once '../../includes/school_context.php';
requireAdminLogin();

if (!isSuperAdmin()) {
    header('Location: ../../admin/dashboard.php');
    exit;
}

$pdo = getDB();

// Handle admin activation/deactivation
if (isset($_GET['toggle_admin'])) {
    $admin_id = (int)$_GET['toggle_admin'];
    $new_status = (int)$_GET['status'];
    
    try {
        $stmt = $pdo->prepare("UPDATE admins SET is_active = ? WHERE id = ?");
        $stmt->execute([$new_status, $admin_id]);
        
        header('Location: list.php?msg=Admin status updated');
        exit;
    } catch (Exception $e) {
        $error = "Error updating admin status: " . $e->getMessage();
    }
}

// Handle school access
if (isset($_GET['access_school'])) {
    $ref = $_GET['access_school'];
    if (setCurrentSchool($ref)) {
        header('Location: ../../admin/dashboard.php');
        exit;
    } else {
        $error = "Failed to access school.";
    }
}

// Get all admins with their school info
$stmt = $pdo->query("
    SELECT 
        a.*,
        ar.academy_name,
        ar.reference_code,
        ar.is_active as school_active,
        aw.balance as wallet_balance,
        aw.total_earnings,
        COUNT(DISTINCT s.id) as student_count
    FROM admins a
    LEFT JOIN academy_references ar ON a.academy_reference = ar.reference_code
    LEFT JOIN admin_wallet aw ON a.id = aw.admin_id
    LEFT JOIN students s ON a.academy_reference = s.academy_reference
    WHERE a.academy_reference IS NOT NULL
    GROUP BY a.id
    ORDER BY a.created_at DESC
");
$admins = $stmt->fetchAll();

// Get super admins
$stmt = $pdo->query("
    SELECT * FROM admins 
    WHERE academy_reference IS NULL 
    ORDER BY username
");
$super_admins = $stmt->fetchAll();
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Manage School Admins - Super Admin</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.3/font/bootstrap-icons.min.css" rel="stylesheet">
    <style>
        :root {
            --primary: #667eea;
            --secondary: #764ba2;
        }
        body {
            background: linear-gradient(135deg, #f5f7fa 0%, #c3cfe2 100%);
        }
        .card {
            border: none;
            border-radius: 1rem;
            box-shadow: 0 4px 15px rgba(0,0,0,0.08);
        }
        .admin-card {
            transition: all 0.3s;
        }
        .admin-card:hover {
            transform: translateY(-5px);
            box-shadow: 0 8px 25px rgba(0,0,0,0.12);
        }
    </style>
</head>
<body>
    <nav class="navbar navbar-dark bg-dark">
        <div class="container-fluid">
            <a href="../dashboard.php" class="navbar-brand">
                <i class="bi bi-arrow-left me-2"></i>Super Admin Dashboard
            </a>
            <span class="navbar-text text-white">
                <i class="bi bi-shield-check me-2"></i>Super Admin
            </span>
        </div>
    </nav>

    <div class="container my-4">
        <!-- Success/Error Messages -->
        <?php if (isset($_GET['msg'])): ?>
            <div class="alert alert-success alert-dismissible fade show">
                <i class="bi bi-check-circle me-2"></i><?= htmlspecialchars($_GET['msg']) ?>
                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
            </div>
        <?php endif; ?>
        
        <?php if (isset($error)): ?>
            <div class="alert alert-danger alert-dismissible fade show">
                <i class="bi bi-exclamation-triangle me-2"></i><?= htmlspecialchars($error) ?>
                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
            </div>
        <?php endif; ?>
        
        <!-- Header -->
        <div class="d-flex justify-content-between align-items-center mb-4">
            <div>
                <h2><i class="bi bi-people-fill me-2"></i>School Administrators</h2>
                <p class="text-muted mb-0">Manage admin accounts for all schools</p>
            </div>
            <a href="create.php" class="btn btn-primary">
                <i class="bi bi-person-plus me-2"></i>Create New Admin
            </a>
        </div>

        <!-- Stats -->
        <div class="row mb-4">
            <div class="col-md-3">
                <div class="card bg-primary text-white">
                    <div class="card-body text-center">
                        <h3><?= count($admins) ?></h3>
                        <p class="mb-0">School Admins</p>
                    </div>
                </div>
            </div>
            <div class="col-md-3">
                <div class="card bg-success text-white">
                    <div class="card-body text-center">
                        <h3><?= count(array_filter($admins, fn($a) => $a['is_active'])) ?></h3>
                        <p class="mb-0">Active</p>
                    </div>
                </div>
            </div>
            <div class="col-md-3">
                <div class="card bg-warning text-white">
                    <div class="card-body text-center">
                        <h3><?= count($super_admins) ?></h3>
                        <p class="mb-0">Super Admins</p>
                    </div>
                </div>
            </div>
            <div class="col-md-3">
                <div class="card bg-info text-white">
                    <div class="card-body text-center">
                        <h3><?= array_sum(array_column($admins, 'student_count')) ?></h3>
                        <p class="mb-0">Total Students</p>
                    </div>
                </div>
            </div>
        </div>

        <!-- Super Admins -->
        <div class="card mb-4">
            <div class="card-header bg-warning text-dark">
                <h5 class="mb-0"><i class="bi bi-shield-check me-2"></i>Super Administrators</h5>
            </div>
            <div class="card-body">
                <div class="table-responsive">
                    <table class="table table-hover">
                        <thead>
                            <tr>
                                <th>Username</th>
                                <th>Email</th>
                                <th>Status</th>
                                <th>Created</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($super_admins as $admin): ?>
                            <tr>
                                <td><strong><?= htmlspecialchars($admin['username']) ?></strong></td>
                                <td><?= htmlspecialchars($admin['email'] ?? 'N/A') ?></td>
                                <td>
                                    <span class="badge bg-success">Super Admin</span>
                                </td>
                                <td><?= date('M d, Y', strtotime($admin['created_at'])) ?></td>
                            </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            </div>
        </div>

        <!-- School Admins -->
        <div class="card">
            <div class="card-header bg-light">
                <h5 class="mb-0"><i class="bi bi-people me-2"></i>School Administrators</h5>
            </div>
            <div class="card-body">
                <?php if (empty($admins)): ?>
                    <div class="text-center py-5">
                        <i class="bi bi-inbox fs-1 text-muted mb-3"></i>
                        <h5 class="text-muted">No school admins yet</h5>
                        <p class="text-muted">Create your first school admin to get started.</p>
                        <a href="create.php" class="btn btn-primary">
                            <i class="bi bi-plus-circle me-2"></i>Create School Admin
                        </a>
                    </div>
                <?php else: ?>
                    <div class="table-responsive">
                        <table class="table table-hover align-middle">
                            <thead>
                                <tr>
                                    <th>Admin</th>
                                    <th>School</th>
                                    <th>Contact</th>
                                    <th>Students</th>
                                    <th>Wallet</th>
                                    <th>Status</th>
                                    <th>Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($admins as $admin): ?>
                                <tr>
                                    <td>
                                        <div>
                                            <strong><?= htmlspecialchars($admin['username']) ?></strong>
                                            <?php if (!empty($admin['full_name'] ?? '')): ?>
                                                <br><small class="text-muted"><?= htmlspecialchars($admin['full_name']) ?></small>
                                            <?php endif; ?>
                                        </div>
                                    </td>
                                    <td>
                                        <strong><?= htmlspecialchars($admin['academy_name'] ?? 'N/A') ?></strong>
                                        <br>
                                        <span class="badge bg-secondary"><?= htmlspecialchars($admin['reference_code'] ?? '') ?></span>
                                    </td>
                                    <td>
                                        <?php if (!empty($admin['email'] ?? '')): ?>
                                            <small><i class="bi bi-envelope me-1"></i><?= htmlspecialchars($admin['email']) ?></small><br>
                                        <?php endif; ?>
                                        <?php if (!empty($admin['phone'] ?? '')): ?>
                                            <small><i class="bi bi-phone me-1"></i><?= htmlspecialchars($admin['phone']) ?></small>
                                        <?php else: ?>
                                            <small class="text-muted">No contact info</small>
                                        <?php endif; ?>
                                    </td>
                                    <td>
                                        <span class="badge bg-info"><?= $admin['student_count'] ?> students</span>
                                    </td>
                                    <td>
                                        <small class="text-muted">Balance:</small>
                                        <br><strong>E<?= number_format($admin['wallet_balance'] ?? 0, 2) ?></strong>
                                        <br><small class="text-success">Earned: E<?= number_format($admin['total_earnings'] ?? 0, 2) ?></small>
                                    </td>
                                    <td>
                                        <?php if ($admin['is_active']): ?>
                                            <span class="badge bg-success">Active</span>
                                        <?php else: ?>
                                            <span class="badge bg-danger">Inactive</span>
                                        <?php endif; ?>
                                        <?php if (!$admin['school_active']): ?>
                                            <br><small class="text-danger">School Inactive</small>
                                        <?php endif; ?>
                                    </td>
                                    <td>
                                        <div class="btn-group btn-group-sm">
                                            <a href="?access_school=<?= urlencode($admin['reference_code']) ?>" 
                                               class="btn btn-outline-primary" 
                                               title="Access This School">
                                                <i class="bi bi-box-arrow-in-right"></i>
                                            </a>
                                            <a href="../schools/edit.php?id=<?= $admin['id'] ?>" 
                                               class="btn btn-outline-secondary" 
                                               title="Edit School">
                                                <i class="bi bi-building"></i>
                                            </a>
                                            <?php if ($admin['is_active']): ?>
                                                <button class="btn btn-outline-warning" 
                                                        onclick="toggleAdmin(<?= $admin['id'] ?>, 0, '<?= htmlspecialchars($admin['username']) ?>')"
                                                        title="Deactivate Admin">
                                                    <i class="bi bi-pause-circle"></i>
                                                </button>
                                            <?php else: ?>
                                                <button class="btn btn-outline-success" 
                                                        onclick="toggleAdmin(<?= $admin['id'] ?>, 1, '<?= htmlspecialchars($admin['username']) ?>')"
                                                        title="Activate Admin">
                                                    <i class="bi bi-play-circle"></i>
                                                </button>
                                            <?php endif; ?>
                                        </div>
                                    </td>
                                </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                <?php endif; ?>
            </div>
        </div>
    </div>
    
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        // Toggle admin active status
        function toggleAdmin(adminId, newStatus, username) {
            const action = newStatus == 1 ? 'activate' : 'deactivate';
            const message = `Are you sure you want to ${action} admin: ${username}?`;
            
            if (confirm(message)) {
                window.location.href = `list.php?toggle_admin=${adminId}&status=${newStatus}`;
            }
        }
    </script>
</body>
</html>
