<?php
/**
 * Platform Earnings Report
 * Shows all platform fees collected across all schools
 */

require_once '../../includes/functions.php';
require_once '../../includes/school_context.php';
requireAdminLogin();

if (!isSuperAdmin()) {
    header('Location: ../../admin/dashboard.php');
    exit;
}

$pdo = getDB();

// Get date filter
$filterMonth = $_GET['month'] ?? date('Y-m');

// Get platform fees summary
$stmt = $pdo->prepare("
    SELECT 
        COUNT(*) as total_payments,
        SUM(amount) as gross_revenue,
        SUM(platform_fee) as total_platform_fees,
        SUM(admin_earnings) as total_admin_earnings
    FROM payments 
    WHERE status = 'completed'
    AND DATE_FORMAT(payment_date, '%Y-%m') = ?
");
$stmt->execute([$filterMonth]);
$summary = $stmt->fetch();

// Get breakdown by school
$stmt = $pdo->prepare("
    SELECT 
        p.academy_reference,
        ar.academy_name,
        COUNT(*) as payment_count,
        SUM(p.amount) as gross_revenue,
        SUM(p.platform_fee) as platform_fees,
        SUM(p.admin_earnings) as admin_earnings
    FROM payments p
    LEFT JOIN academy_references ar ON ar.reference_code = p.academy_reference
    WHERE p.status = 'completed'
    AND DATE_FORMAT(p.payment_date, '%Y-%m') = ?
    GROUP BY p.academy_reference
    ORDER BY platform_fees DESC
");
$stmt->execute([$filterMonth]);
$breakdown = $stmt->fetchAll();

// Get recent platform fee transactions
$stmt = $pdo->prepare("
    SELECT 
        p.*,
        s.full_name as student_name,
        ar.academy_name
    FROM payments p
    LEFT JOIN students s ON s.id = p.student_id
    LEFT JOIN academy_references ar ON ar.reference_code = p.academy_reference
    WHERE p.status = 'completed'
    AND DATE_FORMAT(p.payment_date, '%Y-%m') = ?
    ORDER BY p.payment_date DESC
    LIMIT 50
");
$stmt->execute([$filterMonth]);
$recent_payments = $stmt->fetchAll();

?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Platform Earnings - Super Admin</title>
    
    <link rel="icon" type="image/svg+xml" href="../../images/img-melanesys-logo.svg">
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.0/font/bootstrap-icons.css">
</head>
<body>
    <nav class="navbar navbar-dark bg-dark">
        <div class="container-fluid">
            <a class="navbar-brand" href="../dashboard.php">
                <i class="bi bi-shield-check me-2"></i>Super Admin
            </a>
            <div class="navbar-nav ms-auto">
                <a class="nav-link" href="../dashboard.php">
                    <i class="bi bi-arrow-left me-1"></i>Back
                </a>
            </div>
        </div>
    </nav>

    <div class="container-fluid py-4">
        <div class="row mb-4">
            <div class="col-12">
                <h1 class="h3">
                    <i class="bi bi-graph-up me-2"></i>
                    Platform Earnings Report
                </h1>
                <p class="text-muted">Track platform fees collected from all schools</p>
            </div>
        </div>

        <!-- Month Filter -->
        <div class="row mb-4">
            <div class="col-md-4">
                <form method="GET" class="card shadow-sm">
                    <div class="card-body">
                        <label class="form-label">Filter by Month:</label>
                        <input type="month" name="month" class="form-control" 
                               value="<?= htmlspecialchars($filterMonth) ?>" 
                               onchange="this.form.submit()">
                    </div>
                </form>
            </div>
        </div>

        <!-- Summary Cards -->
        <div class="row g-4 mb-4">
            <div class="col-md-3">
                <div class="card shadow-sm">
                    <div class="card-body">
                        <h6 class="text-muted mb-1">Total Payments</h6>
                        <h2 class="mb-0"><?= number_format($summary['total_payments'] ?? 0) ?></h2>
                    </div>
                </div>
            </div>
            <div class="col-md-3">
                <div class="card shadow-sm">
                    <div class="card-body">
                        <h6 class="text-muted mb-1">Gross Revenue</h6>
                        <h2 class="mb-0">E<?= number_format($summary['gross_revenue'] ?? 0, 2) ?></h2>
                    </div>
                </div>
            </div>
            <div class="col-md-3">
                <div class="card shadow-sm bg-success text-white">
                    <div class="card-body">
                        <h6 class="mb-1 opacity-75">Platform Fees Earned</h6>
                        <h2 class="mb-0">E<?= number_format($summary['total_platform_fees'] ?? 0, 2) ?></h2>
                    </div>
                </div>
            </div>
            <div class="col-md-3">
                <div class="card shadow-sm">
                    <div class="card-body">
                        <h6 class="text-muted mb-1">Admin Earnings</h6>
                        <h2 class="mb-0">E<?= number_format($summary['total_admin_earnings'] ?? 0, 2) ?></h2>
                    </div>
                </div>
            </div>
        </div>

        <!-- Breakdown by School -->
        <div class="row mb-4">
            <div class="col-12">
                <div class="card shadow-sm">
                    <div class="card-header bg-white">
                        <h5 class="mb-0">
                            <i class="bi bi-building me-2"></i>
                            Breakdown by School
                        </h5>
                    </div>
                    <div class="card-body">
                        <div class="table-responsive">
                            <table class="table table-hover">
                                <thead>
                                    <tr>
                                        <th>School</th>
                                        <th>Reference</th>
                                        <th>Payments</th>
                                        <th>Gross Revenue</th>
                                        <th>Platform Fees</th>
                                        <th>Admin Earnings</th>
                                        <th>Fee %</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php if (empty($breakdown)): ?>
                                        <tr>
                                            <td colspan="7" class="text-center text-muted py-4">
                                                No payments found for this period
                                            </td>
                                        </tr>
                                    <?php else: ?>
                                        <?php foreach ($breakdown as $row): ?>
                                        <tr>
                                            <td><strong><?= htmlspecialchars($row['academy_name'] ?? 'Unknown') ?></strong></td>
                                            <td><span class="badge bg-secondary"><?= htmlspecialchars($row['academy_reference']) ?></span></td>
                                            <td><?= number_format($row['payment_count']) ?></td>
                                            <td>E<?= number_format($row['gross_revenue'], 2) ?></td>
                                            <td class="text-success"><strong>E<?= number_format($row['platform_fees'], 2) ?></strong></td>
                                            <td>E<?= number_format($row['admin_earnings'], 2) ?></td>
                                            <td>
                                                <?php 
                                                $feePercent = $row['gross_revenue'] > 0 
                                                    ? ($row['platform_fees'] / $row['gross_revenue']) * 100 
                                                    : 0;
                                                ?>
                                                <span class="badge bg-info"><?= number_format($feePercent, 1) ?>%</span>
                                            </td>
                                        </tr>
                                        <?php endforeach; ?>
                                    <?php endif; ?>
                                </tbody>
                            </table>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <!-- Recent Transactions -->
        <div class="row">
            <div class="col-12">
                <div class="card shadow-sm">
                    <div class="card-header bg-white">
                        <h5 class="mb-0">
                            <i class="bi bi-clock-history me-2"></i>
                            Recent Transactions
                        </h5>
                    </div>
                    <div class="card-body">
                        <div class="table-responsive">
                            <table class="table table-sm">
                                <thead>
                                    <tr>
                                        <th>Date</th>
                                        <th>School</th>
                                        <th>Student</th>
                                        <th>Amount</th>
                                        <th>Platform Fee</th>
                                        <th>Admin Earnings</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php if (empty($recent_payments)): ?>
                                        <tr>
                                            <td colspan="6" class="text-center text-muted py-3">
                                                No transactions found
                                            </td>
                                        </tr>
                                    <?php else: ?>
                                        <?php foreach ($recent_payments as $payment): ?>
                                        <tr>
                                            <td><small><?= date('M d, Y H:i', strtotime($payment['payment_date'])) ?></small></td>
                                            <td><span class="badge bg-secondary"><?= htmlspecialchars($payment['academy_reference']) ?></span></td>
                                            <td><?= htmlspecialchars($payment['student_name'] ?? 'N/A') ?></td>
                                            <td>E<?= number_format($payment['amount'], 2) ?></td>
                                            <td class="text-success">E<?= number_format($payment['platform_fee'], 2) ?></td>
                                            <td>E<?= number_format($payment['admin_earnings'], 2) ?></td>
                                        </tr>
                                        <?php endforeach; ?>
                                    <?php endif; ?>
                                </tbody>
                            </table>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>

