<?php
/**
 * Super Admin - Tenant Storage Details
 */

require_once '../../includes/functions.php';
require_once '../../includes/school_context.php';
require_once '../../includes/tenant_directory_manager.php';

requireAdminLogin();

if (!isSuperAdmin()) {
    header('Location: ../../admin/dashboard.php');
    exit;
}

$tenantId = strtoupper(trim($_GET['tenant_id'] ?? ''));
if ($tenantId === '') {
    header('Location: overview.php?status=missing_tenant');
    exit;
}

$pdo = getDB();
$tenantStmt = $pdo->prepare("
    SELECT 
        ar.*,
        ts_email.setting_value AS contact_email,
        ts_phone.setting_value AS contact_phone
    FROM academy_references ar
    LEFT JOIN tenant_settings ts_email
        ON ts_email.academy_reference = ar.reference_code
       AND ts_email.setting_key = 'school_email'
    LEFT JOIN tenant_settings ts_phone
        ON ts_phone.academy_reference = ar.reference_code
       AND ts_phone.setting_key = 'school_phone'
    WHERE ar.reference_code = ?
    LIMIT 1
");
$tenantStmt->execute([$tenantId]);
$tenant = $tenantStmt->fetch(PDO::FETCH_ASSOC);

if (!$tenant) {
    header('Location: overview.php?status=tenant_not_found');
    exit;
}

$dirManager = new TenantDirectoryManager($pdo);
$quotaMessages = [];
$quotaErrors = [];
$csrfToken = generateCSRFToken();

if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['quota_form'])) {
    if (!validateCSRFToken($_POST['csrf_token'] ?? '')) {
        $quotaErrors[] = 'Security verification failed. Please refresh and try again.';
    } else {
        $value = trim($_POST['quota_value'] ?? '');
        $unit = $_POST['quota_unit'] ?? 'gb';

        if ($value === '' || !is_numeric($value) || $value <= 0) {
            $quotaErrors[] = 'Please enter a positive number for the quota.';
        } else {
            $value = (float)$value;
            $quotaBytes = $unit === 'mb'
                ? $value * 1024 * 1024
                : $value * 1024 * 1024 * 1024;

            $result = $dirManager->updateTenantQuota($tenantId, (int)$quotaBytes);
            if (!empty($result['success'])) {
                header("Location: details.php?tenant_id=" . urlencode($tenantId) . "&status=quota_updated");
                exit;
            } else {
                $quotaErrors[] = $result['error'] ?? 'Failed to update storage quota.';
            }
        }
    }
}

$usage = $dirManager->calculateTenantStorageUsage($tenantId);
$verification = $dirManager->verifyTenantDirectories($tenantId);
$config = $dirManager->getTenantConfig($tenantId) ?: [];

$quotaBytes = $usage['quota_bytes'] ?? $dirManager->getTenantQuota($tenantId);
$selectedQuotaUnit = 'gb';
$displayQuotaValue = number_format($quotaBytes / (1024 * 1024 * 1024), 2, '.', '');

if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['quota_form']) && !empty($quotaErrors)) {
    $displayQuotaValue = trim($_POST['quota_value'] ?? $displayQuotaValue);
    $selectedQuotaUnit = $_POST['quota_unit'] ?? $selectedQuotaUnit;
}

function formatSize($bytes, $precision = 2) {
    if ($bytes <= 0) {
        return '0 B';
    }
    $units = ['B', 'KB', 'MB', 'GB', 'TB'];
    $power = min(floor(log($bytes, 1024)), count($units) - 1);
    $value = $bytes / (1024 ** $power);
    return number_format($value, $precision) . ' ' . $units[$power];
}

function percentOf($part, $total) {
    if ($total <= 0) {
        return 0;
    }
    return round(($part / $total) * 100, 2);
}

$categoryStats = [];
$categoryStmt = $pdo->prepare("
    SELECT 
        file_category,
        COUNT(*) AS file_count,
        SUM(file_size) AS total_size
    FROM tenant_files
    WHERE tenant_id = ? AND is_deleted = 0
    GROUP BY file_category
    ORDER BY total_size DESC
");
$categoryStmt->execute([$tenantId]);
$categoryStats = $categoryStmt->fetchAll(PDO::FETCH_ASSOC);

$recentFilesStmt = $pdo->prepare("
    SELECT 
        original_filename,
        file_category,
        file_size,
        uploaded_by,
        upload_date
    FROM tenant_files
    WHERE tenant_id = ? AND is_deleted = 0
    ORDER BY upload_date DESC
    LIMIT 15
");
$recentFilesStmt->execute([$tenantId]);
$recentFiles = $recentFilesStmt->fetchAll(PDO::FETCH_ASSOC);

$largestFilesStmt = $pdo->prepare("
    SELECT 
        original_filename,
        file_category,
        file_size,
        upload_date
    FROM tenant_files
    WHERE tenant_id = ? AND is_deleted = 0
    ORDER BY file_size DESC
    LIMIT 10
");
$largestFilesStmt->execute([$tenantId]);
$largestFiles = $largestFilesStmt->fetchAll(PDO::FETCH_ASSOC);

$usageData = $usage['usage'] ?? [
    'total' => 0,
    'uploads' => 0,
    'backups' => 0,
    'logs' => 0,
    'temp' => 0,
    'other' => 0,
];

?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?= htmlspecialchars($tenant['academy_name']) ?> - Storage Details</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.1/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.1/font/bootstrap-icons.css" rel="stylesheet">
    <style>
        body {
            background: #f5f7fb;
        }
        .hero-card {
            background: linear-gradient(135deg, #0f172a, #312e81);
            color: #fff;
            border-radius: 18px;
            padding: 2rem;
            margin-bottom: 2rem;
            position: relative;
            overflow: hidden;
        }
        .hero-card::after {
            content: '';
            position: absolute;
            inset: 0;
            background: radial-gradient(circle at top right, rgba(99, 102, 241, 0.35), transparent 55%);
            pointer-events: none;
        }
        .hero-card .badge {
            background: rgba(255, 255, 255, 0.2);
        }
        .metric-card {
            border-radius: 16px;
            border: none;
            box-shadow: 0 12px 24px rgba(15, 23, 42, 0.08);
        }
        .metric-card .icon {
            width: 48px;
            height: 48px;
            display: grid;
            place-items: center;
            border-radius: 12px;
            background: rgba(99, 102, 241, 0.12);
            color: #312e81;
            font-size: 1.5rem;
        }
        .list-strip {
            border-radius: 14px;
        }
        .list-strip .list-group-item {
            border: none;
            border-radius: 12px;
            margin-bottom: 0.5rem;
            background: #f8fafc;
        }
        .stat-dot {
            display: inline-block;
            width: 10px;
            height: 10px;
            border-radius: 50%;
            margin-right: 0.35rem;
        }
        .table-rounded {
            border-radius: 16px;
            overflow: hidden;
        }
    </style>
</head>
<body>
    <nav class="navbar navbar-dark bg-dark">
        <div class="container-fluid">
            <a class="navbar-brand" href="../dashboard.php">
                <i class="bi bi-shield-lock me-2"></i>Super Admin
            </a>
            <div class="d-flex gap-2">
                <a href="overview.php" class="btn btn-outline-light btn-sm">
                    <i class="bi bi-arrow-left"></i> Back to Storage Overview
                </a>
            </div>
        </div>
    </nav>

    <div class="container-fluid py-4">
        <?php if (isset($_GET['status']) && $_GET['status'] === 'quota_updated'): ?>
            <div class="alert alert-success alert-dismissible fade show">
                <i class="bi bi-check-circle me-2"></i>Storage quota updated successfully.
                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
            </div>
        <?php endif; ?>

        <?php if (!empty($quotaErrors)): ?>
            <div class="alert alert-danger alert-dismissible fade show">
                <i class="bi bi-x-circle me-2"></i><?= htmlspecialchars(implode(' ', $quotaErrors)) ?>
                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
            </div>
        <?php endif; ?>

        <div class="hero-card position-relative">
            <div class="d-flex flex-column flex-md-row justify-content-between align-items-start align-items-md-center">
                <div>
                    <span class="badge text-uppercase mb-2"><?= htmlspecialchars($tenantId) ?></span>
                    <h2 class="mb-2"><?= htmlspecialchars($tenant['academy_name']) ?></h2>
                    <div class="text-white-50">
                        <?php if (!empty($tenant['contact_email'])): ?>
                            <i class="bi bi-envelope me-1"></i><?= htmlspecialchars($tenant['contact_email']) ?>
                        <?php endif; ?>
                        <?php if (!empty($tenant['contact_phone'])): ?>
                            <span class="ms-3"><i class="bi bi-telephone me-1"></i><?= htmlspecialchars($tenant['contact_phone']) ?></span>
                        <?php endif; ?>
                    </div>
                </div>
                <div class="text-end mt-3 mt-md-0">
                    <small class="text-white-50 text-uppercase">Total Usage</small>
                    <div class="display-6 fw-semibold">
                        <?= formatSize($usageData['total'] ?? 0) ?>
                    </div>
                    <div class="d-flex align-items-center justify-content-end gap-2">
                        <div class="progress" style="width: 160px; height: 8px;">
                            <div class="progress-bar bg-info" role="progressbar"
                                style="width: <?= min($usage['percentage'] ?? 0, 100) ?>%;"
                                aria-valuenow="<?= $usage['percentage'] ?? 0 ?>"
                                aria-valuemin="0"
                                aria-valuemax="100"></div>
                        </div>
                        <span class="fw-semibold"><?= number_format($usage['percentage'] ?? 0, 2) ?>%</span>
                    </div>
                    <small class="text-white-50">
                        Quota: <?= formatSize($quotaBytes) ?> · Available: <?= formatSize($usage['available_bytes'] ?? max(0, $quotaBytes - ($usageData['total'] ?? 0))) ?>
                    </small>
                </div>
            </div>
        </div>

        <div class="row g-4 mb-4">
            <div class="col-md-3">
                <div class="card metric-card p-3">
                    <div class="d-flex justify-content-between align-items-start">
                        <div>
                            <p class="text-uppercase text-muted mb-1 small">Uploads</p>
                            <h4><?= formatSize($usageData['uploads'] ?? 0) ?></h4>
                        </div>
                        <div class="icon bg-primary bg-opacity-10 text-primary">
                            <i class="bi bi-cloud-arrow-up"></i>
                        </div>
                    </div>
                    <small class="text-muted"><?= number_format(percentOf($usageData['uploads'] ?? 0, $usageData['total'] ?: 1), 2) ?>% of total</small>
                </div>
            </div>
            <div class="col-md-3">
                <div class="card metric-card p-3">
                    <div class="d-flex justify-content-between align-items-start">
                        <div>
                            <p class="text-uppercase text-muted mb-1 small">Backups</p>
                            <h4><?= formatSize($usageData['backups'] ?? 0) ?></h4>
                        </div>
                        <div class="icon bg-success bg-opacity-10 text-success">
                            <i class="bi bi-hdd-network"></i>
                        </div>
                    </div>
                    <small class="text-muted"><?= number_format(percentOf($usageData['backups'] ?? 0, $usageData['total'] ?: 1), 2) ?>% of total</small>
                </div>
            </div>
            <div class="col-md-3">
                <div class="card metric-card p-3">
                    <div class="d-flex justify-content-between align-items-start">
                        <div>
                            <p class="text-uppercase text-muted mb-1 small">Logs</p>
                            <h4><?= formatSize($usageData['logs'] ?? 0) ?></h4>
                        </div>
                        <div class="icon bg-warning bg-opacity-10 text-warning">
                            <i class="bi bi-file-earmark-text"></i>
                        </div>
                    </div>
                    <small class="text-muted"><?= number_format(percentOf($usageData['logs'] ?? 0, $usageData['total'] ?: 1), 2) ?>% of total</small>
                </div>
            </div>
            <div class="col-md-3">
                <div class="card metric-card p-3">
                    <div class="d-flex justify-content-between align-items-start">
                        <div>
                            <p class="text-uppercase text-muted mb-1 small">Temporary</p>
                            <h4><?= formatSize($usageData['temp'] ?? 0) ?></h4>
                        </div>
                        <div class="icon bg-danger bg-opacity-10 text-danger">
                            <i class="bi bi-trash"></i>
                        </div>
                    </div>
                    <small class="text-muted"><?= number_format(percentOf($usageData['temp'] ?? 0, $usageData['total'] ?: 1), 2) ?>% of total</small>
                </div>
            </div>
        </div>

        <div class="row g-4">
            <div class="col-lg-8">
                <div class="card table-rounded shadow-sm h-100">
                    <div class="card-header bg-white d-flex justify-content-between align-items-center">
                        <div>
                            <h5 class="mb-0"><i class="bi bi-collection me-1"></i>File Categories</h5>
                            <small class="text-muted">Breakdown of tracked uploads from the database</small>
                        </div>
                    </div>
                    <div class="card-body">
                        <?php if (empty($categoryStats)): ?>
                            <p class="text-muted text-center mb-0">No tracked files for this tenant yet.</p>
                        <?php else: ?>
                            <div class="table-responsive">
                                <table class="table align-middle">
                                    <thead class="table-light">
                                        <tr>
                                            <th>Category</th>
                                            <th class="text-center">Files</th>
                                            <th class="text-end">Size</th>
                                            <th class="text-end">Share</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php foreach ($categoryStats as $row): ?>
                                        <?php
                                            $category = $row['file_category'] ?: 'uncategorised';
                                            $percentage = percentOf((int)$row['total_size'], $usageData['uploads'] ?: $usageData['total'] ?: 1);
                                        ?>
                                        <tr>
                                            <td>
                                                <span class="stat-dot bg-primary"></span>
                                                <?= htmlspecialchars(ucwords(str_replace('_', ' ', $category))) ?>
                                            </td>
                                            <td class="text-center"><?= number_format($row['file_count']) ?></td>
                                            <td class="text-end"><?= formatSize((int)$row['total_size']) ?></td>
                                            <td class="text-end">
                                                <div class="d-inline-flex align-items-center gap-2">
                                                    <div class="progress" style="width: 120px; height: 6px;">
                                                        <div class="progress-bar bg-primary" role="progressbar" style="width: <?= min($percentage, 100) ?>%;"></div>
                                                    </div>
                                                    <span><?= number_format($percentage, 2) ?>%</span>
                                                </div>
                                            </td>
                                        </tr>
                                        <?php endforeach; ?>
                                    </tbody>
                                </table>
                            </div>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
            <div class="col-lg-4">
                <div class="card shadow-sm h-100">
                    <div class="card-header bg-white">
                        <h5 class="mb-0"><i class="bi bi-diagram-3 me-1"></i>Directory Health</h5>
                    </div>
                    <div class="card-body">
                        <?php if (empty($verification) || !$verification['success']): ?>
                            <div class="alert alert-warning">
                                <i class="bi bi-exclamation-triangle me-1"></i>
                                <?= htmlspecialchars($verification['message'] ?? ($verification['error'] ?? 'Unable to verify tenant directories.')) ?>
                            </div>
                        <?php endif; ?>

                        <?php if (!empty($verification['details'])): ?>
                            <div class="list-group">
                                <div class="list-group-item d-flex justify-content-between align-items-center">
                                    <span>Directories verified</span>
                                    <span class="badge bg-success"><?= $verification['directories_ok'] ?? 0 ?></span>
                                </div>
                                <div class="list-group-item d-flex justify-content-between align-items-center">
                                    <span>Missing directories</span>
                                    <span class="badge bg-danger"><?= $verification['directories_missing'] ?? 0 ?></span>
                                </div>
                                <div class="list-group-item d-flex justify-content-between align-items-center">
                                    <span>Writable</span>
                                    <span class="badge bg-success"><?= $verification['directories_writable'] ?? 0 ?></span>
                                </div>
                                <div class="list-group-item d-flex justify-content-between align-items-center">
                                    <span>Not writable</span>
                                    <span class="badge bg-warning text-dark"><?= $verification['directories_not_writable'] ?? 0 ?></span>
                                </div>
                            </div>
                        <?php endif; ?>

                        <?php if (!empty($config)): ?>
                            <hr>
                            <h6 class="text-muted text-uppercase small">Configuration</h6>
                            <ul class="list-group list-group-flush">
                                <?php foreach ($config as $key => $value): ?>
                                    <?php if ($key === 'last_updated') continue; ?>
                                    <li class="list-group-item px-0 d-flex justify-content-between align-items-center">
                                        <span class="text-muted"><?= htmlspecialchars(ucwords(str_replace('_', ' ', $key))) ?></span>
                                        <span><?= htmlspecialchars(is_scalar($value) ? (string)$value : json_encode($value)) ?></span>
                                    </li>
                                <?php endforeach; ?>
                                <li class="list-group-item px-0 d-flex justify-content-between align-items-center">
                                    <span class="text-muted">Last updated</span>
                                    <span><?= htmlspecialchars($config['last_updated'] ?? 'Not recorded') ?></span>
                                </li>
                            </ul>
                        <?php endif; ?>

                        <hr>
                        <h6 class="text-muted text-uppercase small">Update Storage Quota</h6>
                        <form method="post" class="row g-2 align-items-end mt-1">
                            <input type="hidden" name="quota_form" value="1">
                            <input type="hidden" name="csrf_token" value="<?= htmlspecialchars($csrfToken) ?>">
                            <div class="col-6">
                                <label class="form-label small text-muted mb-1">New quota value</label>
                                <input type="number" name="quota_value" min="0" step="0.1" class="form-control"
                                       placeholder="e.g. 5"
                                       value="<?= htmlspecialchars($displayQuotaValue) ?>">
                            </div>
                            <div class="col-3">
                                <label class="form-label small text-muted mb-1">Unit</label>
                                <select name="quota_unit" class="form-select">
                                    <option value="gb" <?= $selectedQuotaUnit === 'gb' ? 'selected' : '' ?>>GB</option>
                                    <option value="mb" <?= $selectedQuotaUnit === 'mb' ? 'selected' : '' ?>>MB</option>
                                </select>
                            </div>
                            <div class="col-3 d-grid">
                                <button type="submit" class="btn btn-primary">
                                    <i class="bi bi-save me-1"></i>Save
                                </button>
                            </div>
                        </form>
                        <small class="text-muted d-block mt-2">
                            Current quota: <?= formatSize($quotaBytes) ?>
                        </small>
                    </div>
                </div>
            </div>
        </div>

        <div class="row g-4 mt-1">
            <div class="col-lg-6">
                <div class="card shadow-sm h-100">
                    <div class="card-header bg-white d-flex justify-content-between align-items-center">
                        <h5 class="mb-0"><i class="bi bi-clock-history me-1"></i> Recently Uploaded Files</h5>
                        <span class="badge bg-light text-dark"><?= count($recentFiles) ?> listed</span>
                    </div>
                    <div class="card-body">
                        <?php if (empty($recentFiles)): ?>
                            <p class="text-muted text-center mb-0">No uploads recorded yet.</p>
                        <?php else: ?>
                            <div class="list-group list-strip">
                                <?php foreach ($recentFiles as $file): ?>
                                    <div class="list-group-item">
                                        <div class="d-flex justify-content-between align-items-start">
                                            <div>
                                                <strong><?= htmlspecialchars($file['original_filename']) ?></strong>
                                                <div class="text-muted small mt-1">
                                                    <span class="badge bg-secondary">
                                                        <?= htmlspecialchars($file['file_category'] ?: 'uncategorised') ?>
                                                    </span>
                                                    <span class="ms-2"><?= htmlspecialchars($file['uploaded_by'] ?? 'System') ?></span>
                                                </div>
                                            </div>
                                            <div class="text-end">
                                                <div class="fw-semibold"><?= formatSize((int)$file['file_size']) ?></div>
                                                <small class="text-muted"><?= date('Y-m-d H:i', strtotime($file['upload_date'])) ?></small>
                                            </div>
                                        </div>
                                    </div>
                                <?php endforeach; ?>
                            </div>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
            <div class="col-lg-6">
                <div class="card shadow-sm h-100">
                    <div class="card-header bg-white">
                        <h5 class="mb-0"><i class="bi bi-file-earmark-bar-graph me-1"></i> Largest Files</h5>
                    </div>
                    <div class="card-body">
                        <?php if (empty($largestFiles)): ?>
                            <p class="text-muted text-center mb-0">No tracked files found.</p>
                        <?php else: ?>
                            <table class="table table-sm align-middle">
                                <thead class="table-light">
                                    <tr>
                                        <th>Filename</th>
                                        <th class="text-center">Category</th>
                                        <th class="text-end">Size</th>
                                        <th class="text-end">Uploaded</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach ($largestFiles as $file): ?>
                                    <tr>
                                        <td><?= htmlspecialchars($file['original_filename']) ?></td>
                                        <td class="text-center">
                                            <span class="badge bg-secondary">
                                                <?= htmlspecialchars($file['file_category'] ?: 'uncategorised') ?>
                                            </span>
                                        </td>
                                        <td class="text-end"><?= formatSize((int)$file['file_size']) ?></td>
                                        <td class="text-end"><?= date('Y-m-d', strtotime($file['upload_date'])) ?></td>
                                    </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.1/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>

