<?php
/**
 * Test Tenant Backup System
 * 
 * Run: php test_tenant_backup.php
 */

require_once 'config.php';
require_once 'includes/tenant_directory_manager.php';
require_once 'includes/tenant_backup.php';

// Database connection
$conn = new PDO(
    "mysql:host=" . DB_HOST . ";dbname=" . DB_NAME,
    DB_USER,
    DB_PASS
);

// Initialize
$dirManager = new TenantDirectoryManager($conn);
$backup = new TenantBackup($conn, $dirManager);

// Test tenant
$test_tenant_id = 'test_backup_' . date('Ymd_His');

echo "Testing Tenant Backup System\n";
echo str_repeat("=", 60) . "\n";
echo "Test Tenant: {$test_tenant_id}\n";
echo str_repeat("=", 60) . "\n\n";

// Test 1: Initialize tenant
echo "1. Initializing test tenant...\n";
$initResult = $dirManager->initializeTenant($test_tenant_id, 'Test Backup School');
if ($initResult['success']) {
    echo "   ✓ Tenant initialized\n\n";
} else {
    echo "   ✗ Failed: {$initResult['error']}\n";
    exit(1);
}

// Test 2: Database backup
echo "2. Testing database backup...\n";
$dbResult = $backup->backupTenantDatabase($test_tenant_id);
if ($dbResult['success']) {
    echo "   ✓ Database backup created\n";
    echo "   Filename: {$dbResult['filename']}\n";
    echo "   Size: {$dbResult['size_formatted']}\n";
    echo "   Checksum: {$dbResult['checksum']}\n\n";
} else {
    echo "   ✗ Failed: {$dbResult['error']}\n\n";
}

// Test 3: Files backup
echo "3. Testing files backup...\n";
$filesResult = $backup->backupTenantFiles($test_tenant_id);
if ($filesResult['success']) {
    echo "   ✓ Files backup created\n";
    echo "   Filename: {$filesResult['filename']}\n";
    echo "   Size: {$filesResult['size_formatted']}\n";
    echo "   Files: {$filesResult['file_count']}\n\n";
} else {
    echo "   ✗ Failed: {$filesResult['error']}\n\n";
}

// Test 4: Full backup
echo "4. Testing full backup...\n";
$fullResult = $backup->fullTenantBackup($test_tenant_id);
if ($fullResult['success']) {
    echo "   ✓ Full backup created\n";
    echo "   Filename: {$fullResult['filename']}\n";
    echo "   Size: {$fullResult['size_formatted']}\n\n";
} else {
    echo "   ✗ Failed: {$fullResult['error']}\n\n";
}

// Test 5: List backups
echo "5. Listing backups...\n";
$listResult = $backup->listTenantBackups($test_tenant_id);
if ($listResult['success']) {
    echo "   ✓ Found {$listResult['count']} backup(s)\n";
    foreach ($listResult['backups'] as $b) {
        echo "   - {$b['filename']} ({$b['type']}) - {$b['size_formatted']}\n";
    }
    echo "\n";
} else {
    echo "   ✗ Failed: {$listResult['error']}\n\n";
}

// Test 6: Schedule backup
echo "6. Testing backup scheduling...\n";
$scheduleResult = $backup->scheduleTenantBackup($test_tenant_id, 'daily');
if ($scheduleResult['success']) {
    echo "   ✓ Backup scheduled: {$scheduleResult['schedule']}\n\n";
} else {
    echo "   ✗ Failed: {$scheduleResult['error']}\n\n";
}

// Test 7: Cleanup old backups
echo "7. Testing cleanup (keep 2 backups)...\n";
$cleanupResult = $backup->cleanupOldBackups($test_tenant_id, 2);
if ($cleanupResult['success']) {
    echo "   ✓ Cleanup complete\n";
    echo "   Deleted: {$cleanupResult['deleted']}\n";
    echo "   Kept: {$cleanupResult['kept']}\n";
    echo "   Space freed: {$cleanupResult['space_freed_formatted']}\n\n";
} else {
    echo "   ✗ Failed: {$cleanupResult['error']}\n\n";
}

// Test 8: Cleanup test tenant
echo "8. Cleaning up test tenant...\n";
$testPath = $dirManager->getTenantRootPath($test_tenant_id);
if (file_exists($testPath)) {
    function deleteDir($dir) {
        if (!is_dir($dir)) return unlink($dir);
        foreach (scandir($dir) as $item) {
            if ($item == '.' || $item == '..') continue;
            deleteDir($dir . '/' . $item);
        }
        return rmdir($dir);
    }
    
    if (deleteDir($testPath)) {
        echo "   ✓ Test tenant cleaned up\n";
    }
}

echo "\n" . str_repeat("=", 60) . "\n";
echo "All tests complete!\n";
echo str_repeat("=", 60) . "\n";

