<?php
require_once __DIR__ . '/../includes/functions.php';
require_once __DIR__ . '/../includes/curriculum_catalog.php';

/**
 * Updates the subjects catalogue for Grades 4-7 and Forms 1-5
 * with the new subject definitions supplied by curriculum team.
 *
 * - Subject names are stored with the grade appended e.g. "English Language (Grade 4)"
 * - Codes are generated per grade using grade prefix + supplied code/slug (unique per grade)
 * - Existing grade mappings are cleared before inserting the new mappings
 */

$gradeSubjectMap = getCurriculumSubjectMap();

$pdo = getDB();
$pdo->beginTransaction();

try {
    $maxCodeLength = getSubjectCodeMaxLength($pdo);

    $gradeStmt = $pdo->prepare('SELECT id FROM grades WHERE name = ?');
    $deleteMappingsStmt = $pdo->prepare('DELETE FROM grade_subject WHERE grade_id = ?');
    $insertSubjectStmt = $pdo->prepare('
        INSERT INTO subjects (name, code)
        VALUES (?, ?)
        ON DUPLICATE KEY UPDATE name = VALUES(name)
    ');
    $findSubjectIdStmt = $pdo->prepare('SELECT id FROM subjects WHERE code = ?');
    $insertMappingStmt = $pdo->prepare('INSERT IGNORE INTO grade_subject (grade_id, subject_id) VALUES (?, ?)');

    $summary = [];

    foreach ($gradeSubjectMap as $gradeName => $subjects) {
        $gradeStmt->execute([$gradeName]);
        $gradeId = $gradeStmt->fetchColumn();

        if (!$gradeId) {
            throw new RuntimeException("Grade '$gradeName' not found in database.");
        }

        $deleteMappingsStmt->execute([$gradeId]);

        $count = 0;
        $allowedCodes = [];

        foreach ($subjects as $subject) {
            $code = buildCurriculumSubjectCode($gradeName, $subject['name'], $subject['code'] ?? null, $maxCodeLength);
            $nameWithGrade = $subject['name'] . ' (' . $gradeName . ')';

            $insertSubjectStmt->execute([$nameWithGrade, $code]);

            $subjectId = $pdo->lastInsertId();
            if (!$subjectId) {
                $findSubjectIdStmt->execute([$code]);
                $subjectId = $findSubjectIdStmt->fetchColumn();
            }

            if (!$subjectId) {
                throw new RuntimeException("Failed to resolve subject ID for code $code");
            }

            $insertMappingStmt->execute([$gradeId, $subjectId]);
            $count++;
            $allowedCodes[] = $code;
        }

        // Remove legacy subjects that match the grade prefix but are no longer part of the official list
        if (!empty($allowedCodes)) {
            $placeholders = implode(',', array_fill(0, count($allowedCodes), '?'));
            $prefixPattern = curriculumGradePrefix($gradeName) . '%';

            $cleanupMappingsStmt = $pdo->prepare(
                "DELETE gs FROM grade_subject gs
                 INNER JOIN subjects s ON gs.subject_id = s.id
                 WHERE gs.grade_id = ?
                 AND s.code LIKE ?
                 AND s.code NOT IN ($placeholders)"
            );

            $params = array_merge([$gradeId, $prefixPattern], $allowedCodes);
            $cleanupMappingsStmt->execute($params);

            $cleanupSubjectsStmt = $pdo->prepare(
                "DELETE FROM subjects
                 WHERE code LIKE ?
                 AND code NOT IN ($placeholders)"
            );

            $subjectParams = array_merge([$prefixPattern], $allowedCodes);
            $cleanupSubjectsStmt->execute($subjectParams);
        }

        $summary[$gradeName] = $count;
    }

    $pdo->commit();

    echo "✅ Subject catalogue updated successfully\n\n";
    foreach ($summary as $gradeName => $count) {
        echo sprintf(" - %s: %d subjects updated\n", $gradeName, $count);
    }

} catch (Throwable $e) {
    $pdo->rollBack();
    echo "❌ Error updating subjects: " . $e->getMessage() . "\n";
    exit(1);
}


