<?php
require_once __DIR__ . '/../includes/functions.php';

echo "Updating database schema for monthly payment system...\n";

try {
    $pdo = getDB();
    
    // Add new columns to students table
    $alterations = [
        "ALTER TABLE students ADD COLUMN last_payment_date DATETIME NULL",
        "ALTER TABLE students ADD COLUMN next_payment_due DATETIME NULL", 
        "ALTER TABLE students ADD COLUMN payment_cycle_start DATETIME NULL",
        "ALTER TABLE students ADD COLUMN payment_reminder_sent TINYINT(1) DEFAULT 0",
        "ALTER TABLE students ADD COLUMN warning_email_sent TINYINT(1) DEFAULT 0",
        "ALTER TABLE students ADD COLUMN suspension_warning_sent TINYINT(1) DEFAULT 0",
        "ALTER TABLE students ADD COLUMN moodle_status VARCHAR(20) DEFAULT 'active'",
        "ALTER TABLE students ADD COLUMN suspended_date DATETIME NULL"
    ];
    
    foreach ($alterations as $sql) {
        try {
            $pdo->exec($sql);
            echo "✓ Executed: " . substr($sql, 0, 50) . "...\n";
        } catch (PDOException $e) {
            if (strpos($e->getMessage(), 'Duplicate column name') !== false) {
                echo "⚠ Column already exists: " . substr($sql, 0, 50) . "...\n";
            } else {
                echo "✗ Error: " . $e->getMessage() . "\n";
            }
        }
    }
    
    // Create payment_cycles table to track payment history
    $createCyclesTable = "
        CREATE TABLE IF NOT EXISTS payment_cycles (
            id INT AUTO_INCREMENT PRIMARY KEY,
            student_id INT NOT NULL,
            cycle_start DATETIME NOT NULL,
            cycle_end DATETIME NOT NULL,
            amount_due DECIMAL(10,2) NOT NULL,
            amount_paid DECIMAL(10,2) DEFAULT 0.00,
            status ENUM('active', 'paid', 'overdue', 'suspended') DEFAULT 'active',
            payment_date DATETIME NULL,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            FOREIGN KEY (student_id) REFERENCES students(id) ON DELETE CASCADE,
            INDEX idx_student_cycle (student_id, cycle_start),
            INDEX idx_status (status),
            INDEX idx_cycle_end (cycle_end)
        )
    ";
    
    $pdo->exec($createCyclesTable);
    echo "✓ Created payment_cycles table\n";
    
    // Initialize existing students with payment cycle starting from November 1, 2025
    $initDate = '2025-11-01 00:00:00';
    $nextDue = '2025-12-01 00:00:00';
    
    $updateStudents = "
        UPDATE students 
        SET 
            payment_cycle_start = ?,
            next_payment_due = ?,
            last_payment_date = created_at,
            moodle_status = 'active'
        WHERE payment_cycle_start IS NULL
    ";
    
    $stmt = $pdo->prepare($updateStudents);
    $stmt->execute([$initDate, $nextDue]);
    $affected = $stmt->rowCount();
    echo "✓ Initialized {$affected} students with payment cycle starting November 1, 2025\n";
    
    // Create initial payment cycles for all students
    $createInitialCycles = "
        INSERT INTO payment_cycles (student_id, cycle_start, cycle_end, amount_due, status)
        SELECT 
            s.id,
            s.payment_cycle_start,
            s.next_payment_due,
            (SELECT COUNT(*) * " . SUBJECT_PRICE . " FROM student_subject ss WHERE ss.student_id = s.id) as amount,
            'active'
        FROM students s
        WHERE s.payment_cycle_start IS NOT NULL
        AND NOT EXISTS (
            SELECT 1 FROM payment_cycles pc 
            WHERE pc.student_id = s.id 
            AND pc.cycle_start = s.payment_cycle_start
        )
    ";
    
    $pdo->exec($createInitialCycles);
    echo "✓ Created initial payment cycles for all students\n";
    
    echo "\n🎉 Database schema updated successfully!\n";
    echo "📅 All students now have payment cycles starting November 1, 2025\n";
    echo "💰 Next payment due: December 1, 2025\n";
    
} catch (Exception $e) {
    echo "❌ Error: " . $e->getMessage() . "\n";
    exit(1);
}
?>
