<?php
/**
 * System Integration Verification
 * 
 * Checks all multi-tenant file system components are properly integrated
 * 
 * Run: php tools/verify_system_integration.php
 */

// CLI only
if (php_sapi_name() !== 'cli') {
    die("Run from command line only\n");
}

require_once __DIR__ . '/../config.php';

echo "\n";
echo str_repeat("=", 70) . "\n";
echo "MULTI-TENANT FILE SYSTEM - INTEGRATION VERIFICATION\n";
echo str_repeat("=", 70) . "\n\n";

$results = [];
$totalTests = 0;
$passedTests = 0;

// Test 1: Core files exist
echo "1. Checking core files...\n";
$coreFiles = [
    'includes/tenant_filesystem.php',
    'includes/tenant_directory_manager.php',
    'includes/tenant_file_upload.php',
    'includes/tenant_backup.php',
    'serve_file.php',
    'templates/htaccess_templates.php'
];

foreach ($coreFiles as $file) {
    $path = __DIR__ . '/../' . $file;
    $exists = file_exists($path);
    echo "   " . ($exists ? "✓" : "✗") . " {$file}\n";
    $totalTests++;
    if ($exists) $passedTests++;
}
echo "\n";

// Test 2: Upload endpoints exist
echo "2. Checking upload endpoints...\n";
$uploadFiles = [
    'admin/upload_handler.php',
    'admin/students/upload_photo.php',
    'admin/payments/upload_receipt.php',
    'admin/reports/generate_report.php'
];

foreach ($uploadFiles as $file) {
    $path = __DIR__ . '/../' . $file;
    $exists = file_exists($path);
    echo "   " . ($exists ? "✓" : "✗") . " {$file}\n";
    $totalTests++;
    if ($exists) $passedTests++;
}
echo "\n";

// Test 3: Storage dashboards exist
echo "3. Checking storage dashboards...\n";
$dashboardFiles = [
    'super_admin/storage/overview.php',
    'admin/storage/usage.php'
];

foreach ($dashboardFiles as $file) {
    $path = __DIR__ . '/../' . $file;
    $exists = file_exists($path);
    echo "   " . ($exists ? "✓" : "✗") . " {$file}\n";
    $totalTests++;
    if ($exists) $passedTests++;
}
echo "\n";

// Test 4: API endpoints exist
echo "4. Checking API endpoints...\n";
$apiFiles = [
    'api/storage/tenant.php',
    'api/storage/usage.php',
    'api/storage/file.php'
];

foreach ($apiFiles as $file) {
    $path = __DIR__ . '/../' . $file;
    $exists = file_exists($path);
    echo "   " . ($exists ? "✓" : "✗") . " {$file}\n";
    $totalTests++;
    if ($exists) $passedTests++;
}
echo "\n";

// Test 5: Database connection
echo "5. Testing database connection...\n";
try {
    $conn = new PDO(
        "mysql:host=" . DB_HOST . ";dbname=" . DB_NAME,
        DB_USER,
        DB_PASS
    );
    echo "   ✓ Database connection successful\n";
    $totalTests++;
    $passedTests++;
} catch (PDOException $e) {
    echo "   ✗ Database connection failed: " . $e->getMessage() . "\n";
    $totalTests++;
}
echo "\n";

// Test 6: Database tables exist
echo "6. Checking database tables...\n";
$requiredTables = [
    'tenant_files',
    'tenant_file_access_log',
    'tenant_storage_quotas',
    'file_operations_log'
];

if (isset($conn)) {
    foreach ($requiredTables as $table) {
        try {
            $stmt = $conn->query("SHOW TABLES LIKE '{$table}'");
            $exists = $stmt->rowCount() > 0;
            echo "   " . ($exists ? "✓" : "✗") . " {$table}\n";
            $totalTests++;
            if ($exists) $passedTests++;
        } catch (PDOException $e) {
            echo "   ✗ {$table} (error checking)\n";
            $totalTests++;
        }
    }
}
echo "\n";

// Test 7: Tenants directory exists
echo "7. Checking tenants directory...\n";
$tenantsDir = __DIR__ . '/../tenants';
$exists = file_exists($tenantsDir);
$writable = $exists && is_writable($tenantsDir);

echo "   " . ($exists ? "✓" : "✗") . " Directory exists\n";
echo "   " . ($writable ? "✓" : "✗") . " Directory writable\n";
$totalTests += 2;
if ($exists) $passedTests++;
if ($writable) $passedTests++;
echo "\n";

// Test 8: Check if includes work
echo "8. Testing class loading...\n";
$classTests = [
    'TenantDirectoryManager' => 'includes/tenant_directory_manager.php',
    'TenantFileUpload' => 'includes/tenant_file_upload.php',
    'TenantBackup' => 'includes/tenant_backup.php',
    'HtaccessGenerator' => 'templates/htaccess_templates.php'
];

foreach ($classTests as $class => $file) {
    $path = __DIR__ . '/../' . $file;
    if (file_exists($path)) {
        require_once $path;
        $exists = class_exists($class);
        echo "   " . ($exists ? "✓" : "✗") . " {$class}\n";
        $totalTests++;
        if ($exists) $passedTests++;
    }
}
echo "\n";

// Summary
echo str_repeat("=", 70) . "\n";
echo "VERIFICATION SUMMARY\n";
echo str_repeat("=", 70) . "\n";
echo "Tests Passed: {$passedTests}/{$totalTests}\n";
echo "Success Rate: " . round(($passedTests / $totalTests) * 100, 1) . "%\n";

if ($passedTests === $totalTests) {
    echo "\n✓ ALL TESTS PASSED - System is fully integrated!\n";
    exit(0);
} else {
    echo "\n⚠ Some tests failed - review output above\n";
    exit(1);
}

echo str_repeat("=", 70) . "\n\n";

