<?php
/**
 * Email Functions for Skolo-Kine Learning Platform
 */

require_once __DIR__ . '/email_templates.php';

use PHPMailer\PHPMailer\PHPMailer;
use PHPMailer\PHPMailer\Exception as MailException;

/**
 * Send email verification
 */
function sendVerificationEmail($email, $student_name, $verification_token) {
    try {
        $verification_link = PUBLIC_BASE_URL . "/verify-email.php?token=" . urlencode($verification_token);
        $html_content = getEmailVerificationTemplate($student_name, $verification_link);
        
        return sendEmail($email, $student_name, "Verify Your Email - " . APP_NAME, $html_content);
    } catch (Exception $e) {
        error_log("Email verification failed: " . $e->getMessage());
        return false;
    }
}

/**
 * Send welcome email
 */
function sendWelcomeEmail($email, $student_name, $subjects) {
    try {
        $login_url = MOODLE_URL;
        $html_content = getWelcomeEmailTemplate($student_name, $login_url, $subjects);
        
        return sendEmail($email, $student_name, "Welcome to " . APP_NAME . "!", $html_content);
    } catch (Exception $e) {
        error_log("Welcome email failed: " . $e->getMessage());
        return false;
    }
}

/**
 * Send payment confirmation email
 */
function sendPaymentConfirmationEmail($email, $student_name, $amount, $subjects) {
    try {
        $html_content = getPaymentConfirmationTemplate($student_name, $amount, $subjects);
        
        return sendEmail($email, $student_name, "Payment Confirmation - " . APP_NAME, $html_content);
    } catch (Exception $e) {
        error_log("Payment confirmation email failed: " . $e->getMessage());
        return false;
    }
}

/**
 * Send password reset email
 */
function sendPasswordResetEmail($email, $student_name, $reset_token) {
    try {
        $reset_link = PUBLIC_BASE_URL . "/reset-password.php?token=" . urlencode($reset_token);
        $html_content = getPasswordResetTemplate($student_name, $reset_link);
        
        return sendEmail($email, $student_name, "Password Reset - " . APP_NAME, $html_content);
    } catch (Exception $e) {
        error_log("Password reset email failed: " . $e->getMessage());
        return false;
    }
}

/**
 * Core email sending function
 */
function sendEmail($to_email, $to_name, $subject, $html_content, $text_content = null) {
    try {
        $mail = new PHPMailer(true);
        
        // Server settings
        $mail->isSMTP();
        $mail->Host = SMTP_HOST;
        $mail->SMTPAuth = true;
        $mail->Username = SMTP_USERNAME;
        $mail->Password = SMTP_PASSWORD;
        $mail->SMTPSecure = SMTP_ENCRYPTION;
        $mail->Port = SMTP_PORT;
        $mail->CharSet = 'UTF-8';
        
        // Recipients
        $mail->setFrom(SMTP_FROM_EMAIL, SMTP_FROM_NAME);
        $mail->addAddress($to_email, $to_name);
        $mail->addReplyTo(SMTP_FROM_EMAIL, SMTP_FROM_NAME);
        
        // Content
        $mail->isHTML(true);
        $mail->Subject = $subject;
        $mail->Body = $html_content;
        
        if ($text_content) {
            $mail->AltBody = $text_content;
        } else {
            // Generate text version from HTML
            $mail->AltBody = strip_tags($html_content);
        }
        
        $result = $mail->send();
        
        // Log email sending
        logEmailSent($to_email, $subject, $result ? 'SUCCESS' : 'FAILED');
        
        return $result;
        
    } catch (MailException $e) {
        error_log("Email sending failed: " . $e->getMessage());
        logEmailSent($to_email, $subject, 'FAILED', $e->getMessage());
        return false;
    }
}

/**
 * Log email sending activity
 */
function logEmailSent($to_email, $subject, $status, $error_message = null) {
    try {
        $pdo = getDB();
        $stmt = $pdo->prepare("
            INSERT INTO email_logs (to_email, subject, status, error_message, sent_at) 
            VALUES (?, ?, ?, ?, NOW())
        ");
        $stmt->execute([$to_email, $subject, $status, $error_message]);
    } catch (Exception $e) {
        error_log("Failed to log email: " . $e->getMessage());
    }
}

/**
 * Queue email for sending
 */
function queueEmail($to_email, $to_name, $subject, $html_content, $priority = 'NORMAL') {
    try {
        $pdo = getDB();
        $stmt = $pdo->prepare("
            INSERT INTO email_queue (to_email, to_name, subject, html_content, priority, created_at) 
            VALUES (?, ?, ?, ?, ?, NOW())
        ");
        return $stmt->execute([$to_email, $to_name, $subject, $html_content, $priority]);
    } catch (Exception $e) {
        error_log("Failed to queue email: " . $e->getMessage());
        return false;
    }
}

/**
 * Process email queue
 */
function processEmailQueue($limit = 10) {
    try {
        $pdo = getDB();
        
        // Get queued emails
        $stmt = $pdo->prepare("
            SELECT * FROM email_queue 
            WHERE status = 'PENDING' 
            ORDER BY priority DESC, created_at ASC 
            LIMIT ?
        ");
        $stmt->execute([$limit]);
        $emails = $stmt->fetchAll();
        
        $processed = 0;
        $failed = 0;
        
        foreach ($emails as $email) {
            $success = sendEmail(
                $email['to_email'],
                $email['to_name'],
                $email['subject'],
                $email['html_content']
            );
            
            // Update queue status
            $status = $success ? 'SENT' : 'FAILED';
            $stmt = $pdo->prepare("
                UPDATE email_queue 
                SET status = ?, processed_at = NOW() 
                WHERE id = ?
            ");
            $stmt->execute([$status, $email['id']]);
            
            if ($success) {
                $processed++;
            } else {
                $failed++;
            }
        }
        
        return ['processed' => $processed, 'failed' => $failed];
        
    } catch (Exception $e) {
        error_log("Failed to process email queue: " . $e->getMessage());
        return ['processed' => 0, 'failed' => 0];
    }
}

/**
 * Test email configuration
 */
function testEmailConfiguration() {
    try {
        $mail = new PHPMailer(true);
        
        // Server settings
        $mail->isSMTP();
        $mail->Host = SMTP_HOST;
        $mail->SMTPAuth = true;
        $mail->Username = SMTP_USERNAME;
        $mail->Password = SMTP_PASSWORD;
        $mail->SMTPSecure = SMTP_ENCRYPTION;
        $mail->Port = SMTP_PORT;
        $mail->Timeout = 10;
        
        // Test connection
        if ($mail->smtpConnect()) {
            $mail->smtpClose();
            return ['success' => true, 'message' => 'SMTP connection successful'];
        } else {
            return ['success' => false, 'message' => 'SMTP connection failed'];
        }
        
    } catch (Exception $e) {
        return ['success' => false, 'message' => $e->getMessage()];
    }
}

/**
 * Get email statistics
 */
function getEmailStatistics() {
    try {
        $pdo = getDB();
        
        // Total emails sent
        $stmt = $pdo->query("SELECT COUNT(*) as total FROM email_logs");
        $total = $stmt->fetch()['total'];
        
        // Successful emails
        $stmt = $pdo->query("SELECT COUNT(*) as success FROM email_logs WHERE status = 'SUCCESS'");
        $success = $stmt->fetch()['success'];
        
        // Failed emails
        $stmt = $pdo->query("SELECT COUNT(*) as failed FROM email_logs WHERE status = 'FAILED'");
        $failed = $stmt->fetch()['failed'];
        
        // Queue status
        $stmt = $pdo->query("SELECT COUNT(*) as pending FROM email_queue WHERE status = 'PENDING'");
        $pending = $stmt->fetch()['pending'];
        
        return [
            'total' => $total,
            'success' => $success,
            'failed' => $failed,
            'pending' => $pending,
            'success_rate' => $total > 0 ? round(($success / $total) * 100, 2) : 0
        ];
        
    } catch (Exception $e) {
        error_log("Failed to get email statistics: " . $e->getMessage());
        return [
            'total' => 0,
            'success' => 0,
            'failed' => 0,
            'pending' => 0,
            'success_rate' => 0
        ];
    }
}