<?php
/**
 * API Endpoint: Get Storage Usage
 * 
 * GET /api/storage/usage/{tenant_id}
 * 
 * Returns detailed storage usage including file breakdown
 */

header('Content-Type: application/json');

session_start();
require_once '../../config.php';
require_once '../../includes/tenant_directory_manager.php';

// Check authentication
if (!isset($_SESSION['user_id'])) {
    http_response_code(401);
    echo json_encode(['error' => 'Unauthorized']);
    exit;
}

// Get tenant ID
$tenant_id = $_GET['tenant_id'] ?? $_SESSION['academy_reference'];

// Check permission
if ($_SESSION['role'] !== 'super_admin' && $_SESSION['academy_reference'] !== $tenant_id) {
    http_response_code(403);
    echo json_encode(['error' => 'Access denied']);
    exit;
}

try {
    // Database connection
    $conn = new PDO(
        "mysql:host=" . DB_HOST . ";dbname=" . DB_NAME,
        DB_USER,
        DB_PASS,
        [PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION]
    );
    
    // Initialize directory manager
    $dirManager = new TenantDirectoryManager($conn);
    
    // Get storage usage
    $usage = $dirManager->calculateTenantStorageUsage($tenant_id);
    
    if (!$usage['success']) {
        http_response_code(404);
        echo json_encode(['error' => 'Tenant not found']);
        exit;
    }
    
    // Get detailed file statistics
    $stmt = $conn->prepare("
        SELECT 
            file_category,
            COUNT(*) as file_count,
            SUM(file_size) as total_size,
            AVG(file_size) as avg_size,
            MIN(upload_date) as first_upload,
            MAX(upload_date) as last_upload
        FROM tenant_files
        WHERE tenant_id = ? AND is_deleted = 0
        GROUP BY file_category
    ");
    $stmt->execute([$tenant_id]);
    $categoryDetails = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Format category details
    $categories = [];
    foreach ($categoryDetails as $cat) {
        $categories[$cat['file_category']] = [
            'file_count' => (int)$cat['file_count'],
            'total_size' => (int)$cat['total_size'],
            'total_size_formatted' => formatBytes($cat['total_size']),
            'avg_size' => (int)$cat['avg_size'],
            'avg_size_formatted' => formatBytes($cat['avg_size']),
            'first_upload' => $cat['first_upload'],
            'last_upload' => $cat['last_upload']
        ];
    }
    
    // Return comprehensive data
    echo json_encode([
        'success' => true,
        'tenant_id' => $tenant_id,
        'storage' => [
            'total_bytes' => $usage['total_bytes'],
            'total_formatted' => $usage['total_formatted'],
            'quota_bytes' => $usage['quota_bytes'],
            'quota_formatted' => $usage['quota_formatted'],
            'available_bytes' => $usage['available_bytes'],
            'available_formatted' => $usage['available_formatted'],
            'percentage' => $usage['percentage'],
            'breakdown' => $usage['usage']
        ],
        'categories' => $categories,
        'timestamp' => date('Y-m-d H:i:s')
    ]);
    
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode(['error' => $e->getMessage()]);
}

// formatBytes() function is available from includes/security_headers.php
if (!function_exists('formatBytes')) {
    function formatBytes($bytes) {
        $units = ['B', 'KB', 'MB', 'GB'];
        for ($i = 0; $bytes > 1024 && $i < count($units) - 1; $i++) {
            $bytes /= 1024;
        }
        return round($bytes, 2) . ' ' . $units[$i];
    }
}

